package cn.com.duiba.boot.ext.autoconfigure.cloud.netflix.feign.hystrix;

import cn.com.duiba.boot.exception.BizException;
import cn.com.duiba.boot.ext.autoconfigure.core.rpc.RpcContext;
import cn.com.duiba.boot.netflix.feign.hystrix.FeignHystrixCommand;
import cn.com.duiba.boot.netflix.feign.hystrix.HystrixDisabled;
import com.alibaba.fastjson.JSONObject;
import com.netflix.hystrix.exception.HystrixBadRequestException;
import feign.Response;
import feign.RetryableException;
import feign.Util;
import feign.codec.ErrorDecoder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.lang.reflect.Method;
import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Collection;
import java.util.Date;
import java.util.Map;
import java.util.concurrent.RejectedExecutionException;

import static feign.Util.RETRY_AFTER;
import static feign.Util.checkNotNull;
import static java.lang.String.format;
import static java.util.Locale.US;
import static java.util.concurrent.TimeUnit.SECONDS;

/**
 * 自定义Feign错误解码器，把非2**状态的http响应转换为异常。对于IllegalArgumentException或者注解了@HystrixDisabled的异常会包装为HystrixBadRequestException，避免触发熔断/降级
 */
public class FeignErrorDecoder implements ErrorDecoder {

    private static final Logger logger = LoggerFactory.getLogger(FeignErrorDecoder.class);

    private final RetryAfterDecoder retryAfterDecoder = new RetryAfterDecoder();

    @Override
    public Exception decode(String methodKey, Response response) {
        Exception exception;
        try {
            if (response.body() != null) {
                String body = Util.toString(response.body().asReader());
                JSONObject jsonObj = JSONObject.parseObject(body);
                String exceptionClassStr = jsonObj.getString("exception");
                Class<?> exceptionClass = null;
                if(exceptionClassStr != null){
                    try{
                        exceptionClass = Class.forName(exceptionClassStr);
                    } catch (ClassNotFoundException e) {
                        //Ignore
//                        logger.warn("exceptionClass [{}] not found", exceptionClass);
                    }
                }
                if(exceptionClass != null){
                    if(BizException.class.isAssignableFrom(exceptionClass)){//BizException特殊处理
                        exception = new BizException(jsonObj.getString("message")).withCode(jsonObj.getString("code"));//TODO
                    }else {
                        exception = genFeignException(response.status(), methodKey, body);
                    }

                    if(IllegalArgumentException.class.isAssignableFrom(exceptionClass)
                            || exceptionClass.isAnnotationPresent(HystrixDisabled.class)
                            || isInMethodIgnoreExceptions(RpcContext.getContext().getMethod(), exceptionClass)){
                        //不允许hystrix进行熔断、降级的异常用HystrixBadRequestException包装起来。
                        exception = new HystrixBadRequestException(exception.getMessage(), exception);
                    }else if(RejectedExecutionException.class.isAssignableFrom(exceptionClass)){//服务端拒绝执行,可能是线程池满了/服务器忙
                        exception = new RetryableException(exception.getMessage(), new RejectedExecutionException(exception.getMessage(), exception), null);
                    }
                }else{
                    exception = genFeignException(response.status(), methodKey, body);
                }
            }else{
                exception = genFeignException(response.status(), methodKey, null);
            }
        } catch (IOException ignored) {
            exception = genFeignException(response.status(), methodKey, null);
        }

        Date retryAfter = retryAfterDecoder.apply(firstOrNull(response.headers(), RETRY_AFTER));
        if (retryAfter != null) {
            return new RetryableException(exception.getMessage(), exception, retryAfter);
        }
        return exception;
    }

    /**
     * 判断异常是否在熔断器忽略异常列表中.
     * @param method
     * @param exceptionClass
     * @return
     */
    private boolean isInMethodIgnoreExceptions(Method method, Class exceptionClass){
        FeignHystrixCommand commandOfMethod = method.getAnnotation(FeignHystrixCommand.class);
        FeignHystrixCommand commandOfClass = method.getDeclaringClass().getAnnotation(FeignHystrixCommand.class);
        return isInIgnoreExceptions(commandOfMethod, exceptionClass)
                || isInIgnoreExceptions(commandOfClass, exceptionClass);
    }

    private boolean isInIgnoreExceptions(FeignHystrixCommand command, Class exceptionClass){
        if(command != null){
            Class[] classes = command.ignoreExceptions();
            if(classes != null){
                for(Class clazz : classes){
                    if(clazz.isAssignableFrom(exceptionClass)){
                        return true;
                    }
                }
            }
        }
        return false;
    }

    private <T> T firstOrNull(Map<String, Collection<T>> map, String key) {
        if (map.containsKey(key) && !map.get(key).isEmpty()) {
            return map.get(key).iterator().next();
        }
        return null;
    }

    private FeignClientException genFeignException(int status, String methodKey, String responseBody){
        String message = format("status %s reading %s", status, methodKey);
        message += "; content:\n" + responseBody;
        return new FeignClientException(status, message);
    }

    /**
     * Decodes a {@link Util#RETRY_AFTER} header into an absolute date, if possible. <br> See <a
     * href="https://tools.ietf.org/html/rfc2616#section-14.37">Retry-After format</a>
     */
    static class RetryAfterDecoder {

        static final DateFormat
                RFC822_FORMAT =
                new SimpleDateFormat("EEE, dd MMM yyyy HH:mm:ss 'GMT'", US);
        private final DateFormat rfc822Format;

        RetryAfterDecoder() {
            this(RFC822_FORMAT);
        }

        RetryAfterDecoder(DateFormat rfc822Format) {
            this.rfc822Format = checkNotNull(rfc822Format, "rfc822Format");
        }

        protected long currentTimeMillis() {
            return System.currentTimeMillis();
        }

        /**
         * returns a date that corresponds to the first time a request can be retried.
         *
         * @param retryAfter String in <a href="https://tools.ietf.org/html/rfc2616#section-14.37"
         *                   >Retry-After format</a>
         */
        public Date apply(String retryAfter) {
            if (retryAfter == null) {
                return null;
            }
            if (retryAfter.matches("^[0-9]+$")) {
                long deltaMillis = SECONDS.toMillis(Long.parseLong(retryAfter));
                return new Date(currentTimeMillis() + deltaMillis);
            }
            synchronized (rfc822Format) {
                try {
                    return rfc822Format.parse(retryAfter);
                } catch (ParseException ignored) {
                    return null;
                }
            }
        }
    }

}
