package cn.com.duiba.boot.ext.autoconfigure.graceclose;

import ch.qos.logback.classic.Level;
import ch.qos.logback.classic.LoggerContext;
import cn.com.duiba.boot.ext.autoconfigure.web.BootMonitorCheckFilter;
import com.alibaba.dubbo.config.ProtocolConfig;
import com.netflix.discovery.EurekaClient;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.NoSuchBeanDefinitionException;
import org.springframework.boot.SpringApplication;
import org.springframework.boot.SpringApplicationRunListener;
import org.springframework.boot.context.embedded.EmbeddedWebApplicationContext;
import org.springframework.context.ConfigurableApplicationContext;
import org.springframework.core.env.ConfigurableEnvironment;

import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.IdentityHashMap;
import java.util.List;
import java.util.Map;

/**
 * 自动配置优雅停机
 * <br/>
 *
 * 不能优雅停机的原因：各个框架包括spring、dubbo、xmemcached等在初始化时都添加了自己的shutdownhook，而且这些shutdownhook的执行是无序的，所以有可能在dubbo销毁前xmemcached、datasource等已经销毁，导致关闭期间的几秒内大量报错
 * <br/>
 *
 * 解决方法：在spring启动后通过反射移除所有shutdownhook，并添加必要的spring的shutdownhook、添加必要的dubbo清理代码，实现优雅停机
 *
 * Created by wenqi.huang on 2016/12/27.
 */
public class GracefulCloseRunListener implements SpringApplicationRunListener {

    private static final Logger logger = LoggerFactory.getLogger(GracefulCloseRunListener.class);

    private SpringApplication application;

    /**
     * 必须有这个构造函数，否则spring无法初始化该类
     * @param application
     * @param args
     */
    public GracefulCloseRunListener(SpringApplication application, String[] args) {
        this.application = application;
    }

    //@Override //boot新版本没有这个方法，故去掉Override
    public void started() {

    }

    // boot 1.5.2 新版本有这个方法，故要实现
    public void starting() {

    }

    //@Override
    public void environmentPrepared(ConfigurableEnvironment environment) {

    }

    //@Override
    public void contextPrepared(ConfigurableApplicationContext context) {

    }

    //@Override
    public void contextLoaded(ConfigurableApplicationContext context) {

    }

    @Override
    public void finished(final ConfigurableApplicationContext context, Throwable exception) {
        if(context == null || !(context instanceof EmbeddedWebApplicationContext)){
            return;
        }
        try {
            //TODO 切换到springboot，而且没用memcached客户端的时候，可能不再需要这个类了，到时试试
            Class clazz = Class.forName("java.lang.ApplicationShutdownHooks");
            Field field = clazz.getDeclaredField("hooks");
            field.setAccessible(true);
            IdentityHashMap<Thread, Thread> hooks = (IdentityHashMap<Thread, Thread>) field.get(null);

            //清空原有hook,但保留日志的hook
            List<Thread> keys2remove = new ArrayList<>();
            for(Map.Entry<Thread, Thread> entry : hooks.entrySet()){
                String hookClassName = entry.getKey().getClass().getName();
                if(hookClassName.equals("java.util.logging.LogManager$Cleaner")
                        || hookClassName.equals("org.unidal.helper.Threads$Manager$1") //cat
                        || hookClassName.startsWith("org.jacoco.agent.rt")//jacoco,去掉shutdownhook这个会导致sonarqube 任务执行失败
                        || hookClassName.startsWith("org.quartz")){
                    continue;
                }
                logger.debug("remove shotdownhook:{}", hookClassName);
                keys2remove.add(entry.getKey());
            }
            for(Thread t : keys2remove) {
                hooks.remove(t);
            }

            //添加自己的清理hook
            Runtime.getRuntime().addShutdownHook(new Thread() {
                @Override
                public void run() {
                    long start = System.currentTimeMillis();

                    if(!logger.isInfoEnabled()) {
                        LoggerContext loggerContext = (LoggerContext) LoggerFactory.getILoggerFactory();
                        loggerContext.getLogger(GracefulCloseRunListener.class).setLevel(Level.INFO);
                    }

                    logger.info("begin to shutdown server...");

                    //先让http接口 [/monitor/check] 返回服务不可用
                    logger.info("disable /monitor/check");
                    BootMonitorCheckFilter.setInService(false);

                    try {
                        Class.forName("com.netflix.discovery.EurekaClient");
                        EurekaClient eurekaClient = context.getBean(EurekaClient.class);
                        if (eurekaClient != null) {
                            logger.info("unregister from eureka-server");
                            //spring的做法在关闭的时候会在线程中调用register，同时调用shutdown方法，由于这两个的顺序无法控制，所以有时候可能先shutdown了，然后又把服务注册上去了。
                            //这里在销毁spring之前取出这个bean先行销毁，以防止此bug
                            eurekaClient.shutdown();
                        }
                    }catch(NoSuchBeanDefinitionException | ClassNotFoundException e){//EurekaClient 可能部分项目不会引入此类
                        //Ignore
                    }

                    try {
                        Class.forName("com.alibaba.dubbo.config.ProtocolConfig");
                        logger.info("clean up dubbo");
                        ProtocolConfig.destroyAll();//先清理dubbo
                    } catch(ClassNotFoundException | ExceptionInInitializerError e){
                        logger.info("该项目没有引入dubbo，不会执行dubbo清理");
                        // Ignore
                    } catch (Throwable e) {
                        logger.error(e.getMessage(), e);
                    }

                    logger.info("clean up spring");
                    try {
                        //等待6S的动作会在这个方法内部被调用,见GracefulCloseAutoConfiguration
                        context.close();//再清理spring
                    } catch (Throwable e) {
                        logger.error(e.getMessage(), e);
                    }

                    long period = System.currentTimeMillis() - start;
                    logger.info("shutdown server finished successfully，cost：{} ms", period);

                    //最后优雅关闭Logback，关闭之后日志无法输出到文件
                    LoggerContext loggerContext = (LoggerContext) LoggerFactory.getILoggerFactory();
                    loggerContext.stop();
                }
            });
        } catch (IllegalStateException e) {
            // If the VM is already shutting down,
            // We do not need to register shutdownHook.
        } catch (Exception e){
            logger.error(e.getMessage(), e);
        }
    }

}
