package cn.com.duiba.boot.ext.autoconfigure.perftest;

import cn.com.duiba.boot.ext.autoconfigure.core.SpecifiedBeanPostProcessor;
import cn.com.duiba.boot.ext.autoconfigure.core.ttl.TransmittableExecutorBeanPostProcessor;
import com.alibaba.ttl.TransmittableThreadLocal;
import org.springframework.beans.BeansException;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnResource;
import org.springframework.boot.autoconfigure.condition.ConditionalOnWebApplication;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.jdbc.datasource.lookup.AbstractRoutingDataSource;
import org.springframework.web.servlet.HandlerInterceptor;
import org.springframework.web.servlet.config.annotation.InterceptorRegistration;
import org.springframework.web.servlet.config.annotation.InterceptorRegistry;
import org.springframework.web.servlet.config.annotation.WebMvcConfigurerAdapter;
import org.springframework.web.servlet.handler.HandlerInterceptorAdapter;

import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.sql.DataSource;
import java.lang.reflect.Field;
import java.util.List;

/**
 * 线上压力测试自动配置
 *
 * Created by wenqi.huang on 2016/11/24.
 */
@Configuration
@ConditionalOnResource(resources = "/autoconfig/perftest.properties")
@ConditionalOnClass({TransmittableThreadLocal.class})
public class PerfTestAutoConfiguration {

    public static class TransmittableThreadLocalHolder{
        protected static final TransmittableThreadLocal<Boolean> threadLocal2PressureTest = new TransmittableThreadLocal<Boolean>();
    }

    /**
     * 嵌入springmvc的拦截器，处理包含_duibaPerf=1参数的url(或cookie中有_duibaPerf=1)，数据库全部走影子库
     */
    @Configuration
    @ConditionalOnResource(resources = "/autoconfig/perftest.properties")
    @ConditionalOnClass({TransmittableThreadLocal.class})
    @ConditionalOnWebApplication
    public static class WebConfiguration extends WebMvcConfigurerAdapter {

        @Override
        public void addInterceptors(InterceptorRegistry registry) {
            try {
                //这个拦截器一定要在第一位，否则如果其他拦截器里有数据库/rpc操作会出问题
                //目前只能通过反射放到第一位
                Field field = registry.getClass().getDeclaredField("registrations");
                field.setAccessible(true);
                List<InterceptorRegistration> registrations = (List<InterceptorRegistration>)field.get(registry);
                for(InterceptorRegistration r : registrations){//判断已经添加过则不再添加
                    if(r instanceof InternalInterceptorRegistration){
                        return;
                    }
                }

                InternalInterceptorRegistration registration = new InternalInterceptorRegistration(pressureTestInterceptor());
                registration.addPathPatterns("/**");
                if(registrations.isEmpty()) {
                    registrations.add(registration);
                }else {
                    registrations.add(0, registration);
                }

                //registry.addInterceptor(pressureTestInterceptor()).addPathPatterns("/**");
            }catch(Exception e){
                throw new RuntimeException(e);
            }
        }

        private class InternalInterceptorRegistration extends InterceptorRegistration{

            public InternalInterceptorRegistration(HandlerInterceptor interceptor) {
                super(interceptor);
            }
        }

        private HandlerInterceptor pressureTestInterceptor(){

            return new HandlerInterceptorAdapter() {
                @Override
                public boolean preHandle(HttpServletRequest request, HttpServletResponse response, Object o) throws Exception {
                    TransmittableThreadLocalHolder.threadLocal2PressureTest.remove();
                    boolean isTestMode = false;
                    String testInParameter = request.getParameter("_duibaPerf");
                    if(testInParameter != null && ("1".equals(testInParameter) || "true".equals(testInParameter))){
                        isTestMode = true;
                    }else{
                        Cookie[] cookies = request.getCookies();
                        if(cookies != null) {
                            for (Cookie cookie : cookies) {
                                if("_duibaPerf".equals(cookie.getName()) && ("1".equals(cookie.getValue()) || "true".equals(cookie.getValue()))){
                                    isTestMode = true;
                                }
                            }
                        }
                    }

                    if(isTestMode) {
                        TransmittableThreadLocalHolder.threadLocal2PressureTest.set(true);
                    }

                    return true;
                }

                //postHandle不一定会被调用(preHandle抛出异常或return false时)，用afterCompletion进行清理
                @Override
                public void afterCompletion(HttpServletRequest request, HttpServletResponse response, Object handler, Exception ex) throws Exception {
                    TransmittableThreadLocalHolder.threadLocal2PressureTest.remove();
                }
            };
        }

    }

    /**
     * 把所有线程池转换为经过 TransmittableThreadLocal 包装的线程池,关于TransmittableThreadLocal的作用自行搜索
     */
    @Configuration
    @ConditionalOnResource(resources = "/autoconfig/perftest.properties")
    @ConditionalOnClass({TransmittableThreadLocal.class})
    //@ConditionalOnBean({Executor.class})
    //ConditionalOnMissingBean/ConditionalOnBean注解后BeanPostProcessor不会生成，跟用户配置的CompomentScan路径有关系，如果配了cn.com.duiba，就会出这个问题
    //@ConditionalOnMissingBean({TransmittableExecutorBeanPostProcessor.class})
    public static class TransmittableThreadLocalConfigurer{

        @Bean
        public SpecifiedBeanPostProcessor perfTestExecutorServiceBeanPostProcessor(){
            return new TransmittableExecutorBeanPostProcessor();
        }

    }

    /**
     * 把spring上下文中的所有数据源转换为自动路由的数据源，可以自动把压测流量导入影子库。
     */
    @Configuration
    @ConditionalOnResource(resources = "/autoconfig/perftest.properties")
    @ConditionalOnClass({AbstractRoutingDataSource.class,TransmittableThreadLocal.class})
    //@ConditionalOnBean({DataSource.class})
    public static class DataSourceConfigurer{

        @Bean
        public SpecifiedBeanPostProcessor perfTestDataSourcePostProcessor(){
            return new SpecifiedBeanPostProcessor<DataSource>() {
                @Override
                public int getOrder() {
                    return -2;
                }

                @Override
                public Class<DataSource> getBeanType() {
                    return DataSource.class;
                }

                @Override
                public Object postProcessBeforeInitialization(DataSource bean, String beanName) throws BeansException {
                    return bean;
                }

                @Override
                public Object postProcessAfterInitialization(DataSource bean, String beanName) throws BeansException {
                    if(!(bean instanceof PerfTestRoutingDataSource)){
                        PerfTestRoutingDataSource ts = new PerfTestRoutingDataSource(bean);
                        ts.afterPropertiesSet();
                        bean = ts;
                    }

                    return bean;
                }
            };
        }

    }

}
