package cn.com.duiba.boot.ext.autoconfigure.cat;

import java.io.IOException;
import java.util.*;
import java.util.concurrent.Executor;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.ThreadPoolExecutor;

import javax.annotation.PostConstruct;
import javax.servlet.*;
import javax.servlet.http.HttpServletRequest;

import cn.com.duiba.boot.ext.autoconfigure.cat.annotation.CatTransactionAspect;
import cn.com.duiba.boot.ext.autoconfigure.cloud.netflix.feign.CustomRequestInterceptor;
import cn.com.duiba.boot.ext.autoconfigure.core.SpecifiedBeanPostProcessor;
import cn.com.duiba.boot.ext.autoconfigure.threadpool.wrapper.ScheduledThreadPoolExecutorWrapper;
import cn.com.duiba.boot.ext.autoconfigure.threadpool.wrapper.ThreadPoolExecutorWrapper;
import cn.com.duiba.catmonitor.CatInstance;
import com.alibaba.dubbo.common.Constants;
import com.alibaba.dubbo.common.extension.ExtensionLoader;
import com.alibaba.dubbo.common.store.DataStore;
import com.alibaba.dubbo.registry.dubbo.DubboRegistry;
import com.dianping.cat.status.StatusExtension;
import com.dianping.cat.status.StatusExtensionRegister;
import net.rubyeye.xmemcached.MemcachedClient;

import org.apache.catalina.startup.Tomcat;
import org.apache.ibatis.plugin.Interceptor;
import org.apache.ibatis.session.SqlSessionFactory;
import org.aspectj.lang.annotation.Aspect;
import org.codehaus.plexus.personality.plexus.lifecycle.phase.Initializable;
import org.codehaus.plexus.personality.plexus.lifecycle.phase.InitializationException;
import org.mybatis.spring.SqlSessionFactoryBean;
import org.mybatis.spring.SqlSessionTemplate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeansException;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnWebApplication;
import org.springframework.boot.context.embedded.EmbeddedServletContainer;
import org.springframework.boot.context.embedded.EmbeddedServletContainerInitializedEvent;
import org.springframework.boot.context.embedded.tomcat.TomcatEmbeddedServletContainer;
import org.springframework.boot.context.embedded.tomcat.TomcatEmbeddedServletContainerFactory;
import org.springframework.boot.web.servlet.FilterRegistrationBean;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationListener;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.EnableAspectJAutoProxy;
import org.springframework.context.annotation.Import;
import org.springframework.context.event.ContextRefreshedEvent;
import org.springframework.core.Ordered;
import org.springframework.data.elasticsearch.core.ElasticsearchOperations;
import org.springframework.data.mongodb.core.MongoOperations;
import org.springframework.data.redis.connection.jedis.JedisConnectionFactory;
import org.springframework.util.StringUtils;
import redis.clients.jedis.Jedis;
import cn.com.duiba.catmonitor.mybatis.CatMybatisPlugin;
import cn.com.duiba.wolf.cache.XMemcacheClient;
import cn.com.duiba.wolf.redis.RedisClient;

import com.dianping.cat.Cat;
import com.dianping.cat.servlet.CatFilter;

/**
 * Created by wenqi.huang on 2016/11/7.
 */
@Configuration
@EnableAspectJAutoProxy(proxyTargetClass = true)
@Import(SpringMvcConfigForCat.class)
public class CatAutoConfiguration {
    private static final Logger logger = LoggerFactory.getLogger(CatAutoConfiguration.class);
    private static final String ThreadPoolWaterLevelId = "线程池水位图";

    //ConditionalOnClass这个注解必须加在类上，如果加在方法上的话，会因为找不到这些类而报错：https://github.com/spring-projects/spring-boot/issues/1733
    @Configuration
    @ConditionalOnClass({Servlet.class, FilterRegistrationBean.class, CatFilter.class})
    @ConditionalOnWebApplication
    public static class CatHttpFilterConfiguration{
        /**
         * 注入cat http监控的Filter
         * @return
         */
        @Bean
        public FilterRegistrationBean catHttpFilterConfigurer(){
            CatFilter catFilter = new CatFilter(){
                @Override
                public void doFilter(ServletRequest request, ServletResponse response, FilterChain chain) throws IOException, ServletException {
                    HttpServletRequest req = (HttpServletRequest)request;

                    if("true".equals(req.getHeader(CustomRequestInterceptor.X_RPC))){
                        //不处理RPC请求（来自FeignClient的rest rpc调用）
                        chain.doFilter(request, response);
                        return;
                    }

                    String uri = req.getRequestURI();
                    int index = uri.indexOf('.');
                    if(index > 0){
                        String postfix = uri.substring(index).toLowerCase();
                        if(postfix.equals(".js") || postfix.equals(".gif")//不处理静态资源
                                || postfix.equals(".jpg")
                                || postfix.equals(".png")
                                || postfix.equals(".css")
                                || postfix.equals(".ico")
                                || postfix.equals(".eot")
                                || postfix.equals(".woff")
                                || postfix.equals(".ttf")
                                || postfix.equals(".svg")){
                            chain.doFilter(request, response);
                            return;
                        }
                    }

                    //如果不是前端资源，则继续cat的处理流程
                    super.doFilter(request, response, chain);
                }
            };
            FilterRegistrationBean registrationBean = new FilterRegistrationBean();
            registrationBean.setFilter(catFilter);
            List<String> urlPatterns=new ArrayList<String>();
            urlPatterns.add("/*");//拦截路径，可以添加多个
            registrationBean.setUrlPatterns(urlPatterns);
            registrationBean.setDispatcherTypes(EnumSet.of(DispatcherType.REQUEST, DispatcherType.FORWARD));
            registrationBean.setOrder(1);
            return registrationBean;
        }
    }

    @Configuration
    @ConditionalOnClass({SqlSessionTemplate.class, SqlSessionFactoryBean.class, SqlSessionFactory.class,CatMybatisPlugin.class, Cat.class})
    public static class MyBatisPostProcessorConfiguration{

        /**
         * 声明后置处理器，spring全部bean初始化完成后调用，给所有SqlSessionBean注入CatMybatisPlugin plugin，监控sql的执行
         * @return
         */
        @Bean
        public SpecifiedBeanPostProcessor myBatisPostProcessorConfigurer(){
            return new SpecifiedBeanPostProcessor<Object>() {
                @Override
                public int getOrder() {
                    return -1;
                }

                @Override
                public Class<Object> getBeanType() {
                    return Object.class;
                }

                @Override
                public Object postProcessBeforeInitialization(Object bean, String beanName) throws BeansException {
                    return bean;
                }

                @Override
                public Object postProcessAfterInitialization(Object bean, String beanName) throws BeansException {
                	SqlSessionFactory s = null;
                    if(bean instanceof SqlSessionFactory){
                        s = (SqlSessionFactory)bean;
                    }
                    if(bean instanceof SqlSessionTemplate){
                        s = ((SqlSessionTemplate)bean).getSqlSessionFactory();
                    }
                    if(s == null){
                        return bean;
                    }

                    boolean hasCatPlugin = false;
                    if(s.getConfiguration().getInterceptors() != null && !s.getConfiguration().getInterceptors().isEmpty()) {
                        for (Interceptor plugin : s.getConfiguration().getInterceptors()) {
                            if (plugin instanceof CatMybatisPlugin) {
                                hasCatPlugin = true;
                                break;
                            }
                        }
                    }else{
                    }

                    if (!hasCatPlugin) {
                        s.getConfiguration().addInterceptor(new CatMybatisPlugin());
                    }

                    return bean;
                }
            };
        }
    }
    
    @Configuration
    @ConditionalOnClass({RedisClient.class, Jedis.class, Cat.class, Aspect.class})
    public static class CatRediscacheConfiguration{
    	
    	@Bean
    	public CatRediscachePlugin getCatCachePlugin(){
    		return new CatRediscachePlugin();
    	}
    	
    }

    /**
     * 加入aop，监控spring-data-redis执行耗时
     */
    @Configuration
    @ConditionalOnClass({JedisConnectionFactory.class, Jedis.class, Cat.class, Aspect.class})
    public static class CatSpringDataRedisAspectConfiguration{

    	@Bean
    	public CatSpringDataRedisPlugin catSpringDataRedisAspectPlugin(){
    		return new CatSpringDataRedisPlugin();
    	}

    }

    @Configuration
    @ConditionalOnClass({XMemcacheClient.class, MemcachedClient.class, Cat.class, Aspect.class})
    public static class CatMemcacheConfiguration{
    	
    	@Bean
    	public CatMemcachePlugin getCatMemcachePlugin(){
    		return new CatMemcachePlugin();
    	}
    	
    }
    
    @Configuration
    @ConditionalOnClass({MongoOperations.class, Cat.class, Aspect.class})
    public static class CatMongodbConfiguration{
    	
    	@Bean
    	public CatSpringDataMongodbPlugin getCatSpringDataMongodbPlugin(){
    		return new CatSpringDataMongodbPlugin();
    	}
    	
    }

    @Bean
    public CatTransactionAspect catTransactionAspect(){
        return new CatTransactionAspect();
    }
    
    @Configuration
    @ConditionalOnClass({ElasticsearchOperations.class, Cat.class, Aspect.class})
    public static class CatElasticsearchConfiguration{
    	
    	@Bean
    	public CatSpringDataElasticSearchPlugin getCatSpringDataElasticSearchPlugin(){
    		return new CatSpringDataElasticSearchPlugin();
    	}
    	
    }
    
    @Configuration
    @ConditionalOnClass({Cat.class})
    public static class InitCatServer implements ApplicationListener<ContextRefreshedEvent>, Ordered{

		@Override
		public void onApplicationEvent(ContextRefreshedEvent event) {
			//初始化cat监控
            String appName = event.getApplicationContext().getEnvironment().getProperty("spring.application.name");
            if(StringUtils.isEmpty(appName)){
                logger.warn("you need to set property[spring.application.name] in bootstrap.properties");
                return;
            }
            System.setProperty("cat.app.name", appName);//注入appName,以方便cat获取
			CatInstance.isEnable();
		}

        @Override
        public int getOrder() {
            return -3;
        }
    }

    /**
     * 监控dubbo线程池使用情况，给cat发送心跳来记录
     */
    @Configuration
    @ConditionalOnClass({DubboRegistry.class, DataStore.class, CatFilter.class})
    public static class DubboThreadPoolWaterLevelHeartbeatExtenstion implements StatusExtension, Initializable {

        //给当前的类定义一个extension id, cat服务端会将一个extension id作为一个group，然后将每个key指标都会做图形处理，结果会在cat的heartbeat中进行展示
        @Override
        public String getId() {
            return ThreadPoolWaterLevelId;
        }

        //一个简单描述
        @Override
        public String getDescription() {
            return "各个线程池的水位监控心跳";
        }

        //每隔一分钟自动被cat调用一次
        @Override
        public Map<String, String> getProperties() {
            Map<String, String> maps = new HashMap<>();

            DataStore dataStore = ExtensionLoader.getExtensionLoader(DataStore.class).getDefaultExtension();
            Map<String, Object> executors = dataStore.get(Constants.EXECUTOR_SERVICE_COMPONENT_KEY);

            for(Map.Entry<String, Object> entry : executors.entrySet()) {
                String port = entry.getKey();
                ExecutorService executor = (ExecutorService) entry.getValue();

                if (executor != null && executor instanceof ThreadPoolExecutor) {
                    ThreadPoolExecutor tp = (ThreadPoolExecutor) executor;

                    //这里的每个key指标都会做图形处理，结果会在cat的heartbeat中进行展示,注意，只有value值可以被转化double类型的才会在heartbeat做图形展示
                    String portStr = executors.size() == 1 ? "" : ("port:" + port) + ",";
                    maps.put("Dubbo-ActiveCount("+portStr+"core:"+tp.getCorePoolSize()+",max:"+tp.getMaximumPoolSize()+")", String.valueOf(tp.getActiveCount()));
                }
            }

            return maps;
        }

        //这里是实现了初始化方法，把这个实现注册到cat上，如果你使用spring，需要在spring里面注册此bean，并实现初始化方法。
        @Override
        @PostConstruct
        public void initialize() throws InitializationException {
            StatusExtensionRegister.getInstance().register(this);
        }

    }

    /**
     * 监控Tomcat http线程池使用情况，给cat发送心跳来记录
     */
    @Configuration
    @ConditionalOnWebApplication
    @ConditionalOnClass({TomcatEmbeddedServletContainerFactory.class, Tomcat.class, Servlet.class, CatFilter.class})
    public static class HttpThreadPoolWaterLevelHeartbeatExtenstion implements StatusExtension, Initializable, ApplicationListener<EmbeddedServletContainerInitializedEvent> {

        private ThreadPoolExecutor httpThreadPool;

        //给当前的类定义一个extension id, cat服务端会将一个extension id作为一个group，然后将每个key指标都会做图形处理，结果会在cat的heartbeat中进行展示
        @Override
        public String getId() {
            return ThreadPoolWaterLevelId;
        }

        //一个简单描述
        @Override
        public String getDescription() {
            return "各个线程池的水位监控心跳";
        }

        //每隔一分钟自动被cat调用一次
        @Override
        public Map<String, String> getProperties() {
            Map<String, String> maps = new HashMap<>();

            if(httpThreadPool != null) {
                maps.put("Http-ActiveCount(core:" + httpThreadPool.getCorePoolSize() + ",max:" + httpThreadPool.getMaximumPoolSize() + ")", String.valueOf(httpThreadPool.getActiveCount()));
            }

            return maps;
        }

        //这里是实现了初始化方法，把这个实现注册到cat上，如果你使用spring，需要在spring里面注册此bean，并实现初始化方法。
        @Override
        @PostConstruct
        public void initialize() throws InitializationException {
        }

        @Override
        public void onApplicationEvent(EmbeddedServletContainerInitializedEvent event) {
            EmbeddedServletContainer c = event.getEmbeddedServletContainer();

            if(!(c instanceof TomcatEmbeddedServletContainer)){
                return;
            }
            TomcatEmbeddedServletContainer container = (TomcatEmbeddedServletContainer)c;

            Executor es = container.getTomcat().getConnector().getProtocolHandler().getExecutor();
            if(!(es instanceof ThreadPoolExecutor)) {
                return;
            }
            httpThreadPool = (ThreadPoolExecutor)es;
            //基本的校验通过才注册到cat回调
            StatusExtensionRegister.getInstance().register(this);
        }
    }

    /**
     * 监控DuibaBiz线程池使用情况，给cat发送心跳来记录
     */
    @Configuration
    @ConditionalOnClass({CatFilter.class})
    public static class DubiaBizThreadPoolWaterLevelHeartbeatExtenstion implements StatusExtension, Initializable, ApplicationListener<ContextRefreshedEvent> {

        private ThreadPoolExecutorWrapper executorService;
        private ScheduledThreadPoolExecutorWrapper scheduledExecutorService;

        //给当前的类定义一个extension id, cat服务端会将一个extension id作为一个group，然后将每个key指标都会做图形处理，结果会在cat的heartbeat中进行展示
        @Override
        public String getId() {
            return ThreadPoolWaterLevelId;
        }

        //一个简单描述
        @Override
        public String getDescription() {
            return "各个线程池的水位监控心跳";
        }

        //每隔一分钟自动被cat调用一次
        @Override
        public Map<String, String> getProperties() {
            Map<String, String> maps = new HashMap<>();
            if(executorService != null){
                maps.put("DuibaBiz-ActiveCount(core:"+executorService.getCorePoolSize()+",max:"+executorService.getMaximumPoolSize()+")", String.valueOf(executorService.getActiveCount()));
            }
            if(scheduledExecutorService != null){
                maps.put("DuibaBizScheduled-ActiveCount(core:"+scheduledExecutorService.getCorePoolSize()+",max:"+scheduledExecutorService.getMaximumPoolSize()+")", String.valueOf(scheduledExecutorService.getActiveCount()));
            }

            return maps;
        }

        //这里是实现了初始化方法，把这个实现注册到cat上，如果你使用spring，需要在spring里面注册此bean，并实现初始化方法。
        @Override
        @PostConstruct
        public void initialize() throws InitializationException {
        }

        @Override
        public void onApplicationEvent(ContextRefreshedEvent event) {
            ApplicationContext applicationContext = event.getApplicationContext();

            try {
                Object executorService = applicationContext.getBean("executorService");
                if(executorService != null && executorService instanceof ThreadPoolExecutorWrapper){
                    this.executorService = (ThreadPoolExecutorWrapper)executorService;
                }
            }catch(Exception e){
                //Ignore
            }
            try{
                Object scheduledExecutorService = applicationContext.getBean("scheduledExecutorService");
                if(scheduledExecutorService != null && scheduledExecutorService instanceof ScheduledThreadPoolExecutorWrapper){
                    this.scheduledExecutorService = (ScheduledThreadPoolExecutorWrapper)scheduledExecutorService;
                }
            }catch(Exception e){
                //Ignore
            }

            if(this.executorService != null || this.scheduledExecutorService != null){
                StatusExtensionRegister.getInstance().register(this);
            }
        }
    }

}
