package cn.com.duiba.boot.ext.autoconfigure.cloud.netflix.eureka;

import com.netflix.appinfo.ApplicationInfoManager;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.AutoConfigureAfter;
import org.springframework.boot.autoconfigure.condition.ConditionalOnBean;
import org.springframework.boot.autoconfigure.condition.SearchStrategy;
import org.springframework.cloud.context.scope.refresh.RefreshScopeRefreshedEvent;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.event.EventListener;
import org.springframework.core.Ordered;
import org.springframework.core.annotation.Order;

import javax.annotation.PostConstruct;
import java.util.HashMap;
import java.util.Map;

/**
 * 注册一些额外的元数据到Eureka（Consul）中，比如服务器启动时间。
 */
@Configuration
@AutoConfigureAfter(name = "org.springframework.cloud.netflix.eureka.EurekaClientAutoConfiguration")
@ConditionalOnBean(value={ApplicationInfoManager.class}, search = SearchStrategy.CURRENT)
public class DiscoveryMetadataAutoConfiguration {
    //服务器启动时间
    private static long serverStartUpTime;
    //metadata中的key
    public static final String SERVER_START_UP_TIME_KEY = "serverStartUpTime";
    public static final String WEIGHT_KEY = "weight";

    //applicationInfoManager这个bean有可能是refreshScope的，refresh之后bean实例会变化(实际上拿到的是一个不变的Proxy，实际的会变的对象封装在内部)，这里能确保每次获取这个bean都拿到最新的
    @Autowired
    private ApplicationInfoManager applicationInfoManager;

    /**
     * 获取服务启动时间，确保只生成一次。
     * @return
     */
    private synchronized long getServerStartUpTime(){
        if(serverStartUpTime == 0){
            serverStartUpTime = System.currentTimeMillis();
        }
        return serverStartUpTime;
    }

    //applicationInfoManager这个bean有可能是refreshScope的,所以需要监听refresh事件，监听到变化后重新设置metadata信息。
    //@ConditionalOnClass(RefreshScopeRefreshedEvent.class)?
    @EventListener(value = {RefreshScopeRefreshedEvent.class})
    @Order(Ordered.HIGHEST_PRECEDENCE)
    public void onRefresh(RefreshScopeRefreshedEvent event){
        registerAppMetadataCommon();
    }

    @PostConstruct
    public void init(){
        registerAppMetadataCommon();
    }

    private void registerAppMetadataCommon(){
        long serverStartUpTime = getServerStartUpTime();

        if(applicationInfoManager != null) {
            Map<String, String> metadata = new HashMap<>();
            //metadata中放入服务器启动时间
            metadata.put(SERVER_START_UP_TIME_KEY, String.valueOf(serverStartUpTime));

            //metadata中放入服务器权重,默认100
            //TODO 这里有点小问题，如果有人改了权重，refresh之后权重会被重置回100,后期改掉,把100作为变量放在某处，允许改变
            metadata.put(WEIGHT_KEY, String.valueOf(100));

            applicationInfoManager.registerAppMetadata(metadata);
        }
    }

}
