package cn.com.duiba.boot.ext.autoconfigure.web.mvc;

import cn.com.duiba.boot.ext.autoconfigure.cloud.netflix.feign.CustomSpringMvcContract;
import cn.com.duiba.boot.netflix.feign.AdvancedFeignClient;
import com.alibaba.fastjson.JSON;
import org.apache.commons.lang.ArrayUtils;
import org.springframework.core.CollectionFactory;
import org.springframework.core.MethodParameter;
import org.springframework.web.bind.support.WebDataBinderFactory;
import org.springframework.web.context.request.NativeWebRequest;
import org.springframework.web.method.support.HandlerMethodArgumentResolver;
import org.springframework.web.method.support.ModelAndViewContainer;

import java.lang.reflect.*;
import java.util.Arrays;
import java.util.Collection;

/**
 * AdvancedFeignClient对应的参数解析器。
 * <br/>
 * 这个解析器会在参数所属类的接口注解了@AdvancedFeignClient时生效，把所有的参数按照json格式来反序列化到参数上。
 */
public class AdvancedFeignClientArgumentResolver implements HandlerMethodArgumentResolver {

    @Override
    public boolean supportsParameter(MethodParameter parameter) {
        Class[] interfaces = parameter.getContainingClass().getInterfaces();
        for(Class clazz : interfaces){
            //有接口注解了AdvancedFeignClient则使用当前的mvc参数解析器
            if(clazz.isAnnotationPresent(AdvancedFeignClient.class)){
                return true;
            }
        }
        return false;
    }

    @Override
    public Object resolveArgument(MethodParameter parameter, ModelAndViewContainer mavContainer, NativeWebRequest webRequest, WebDataBinderFactory binderFactory) throws Exception {
        int parameterIndex = parameter.getParameterIndex();
        String parameteyKeyInRequest = CustomSpringMvcContract.HTTP_PARAMETER_PREFIX + parameterIndex;//按参数顺序绑定，第一个参数的key为_p0，第二个为_p1，以此类推
        String[] vals = webRequest.getParameterValues(parameteyKeyInRequest);

        //TODO 这里可能要用getNestedGenericParameterType
        Type parameterType = parameter.getGenericParameterType();
        boolean isIterableType = isParamIterableType(parameter.getParameterType());

        boolean isCollectionSplitted = false;//表示java.lang.Iterable类型是否会被分拆为多个独立的json值，这是FeignClient的处理策略。比如Arrays.asList(new String[]{"1","2"})会被拆分为_p0=1&_p0=2 (_p0表示这是第一个参数)
        if("1".equals(webRequest.getHeader(CustomSpringMvcContract.HTTP_HEADER_PARAM_COLLECTION_SPLITTED))){
            //http头里有这个则明确制定了Collection会被分成多个值,如果是来自代码中的AdvancedFeignClient调用，会自动加上这个头；如果是来自自动化测试等请求，不带这个头则把Collection序列化成一个完整的json数组
            isCollectionSplitted = true;
        }

        if(ArrayUtils.isEmpty(vals)){
            if(isIterableType){
                return createCollection(parameter, vals, isCollectionSplitted);//这里有个小问题，即使客户端调用时List是空的，服务端也会生成空的List(这是因为Feign对List做了特殊处理，无论是null还是空List，都不会把该参数传递到服务端，服务端无法识别，保险起见只能设置为空List了。)
            }else {
                return null;
            }
        }

        try {
            if(!isIterableType){
                if(vals.length == 1){
                    //使用json反序列化。对于String、int等基本类型也能很好地支持
                    return JSON.parseObject(vals[0], parameterType);
                }else{
                    //有多个值，对应的参数类型却不是Collection，则无法处理，抛出异常
                    throw genIllegalArgumentException(isCollectionSplitted, parameter, vals);
                }
            }else{
                if(isCollectionSplitted || (!isCollectionSplitted && vals.length > 1)){//Collection被分割了，则针对内部元素一个个处理并租装起来 (!isCollectionSplitted && vals.length > 1 是防御性写法，虽然告诉我Collection没有分割，但是实际上分割了，也按照分割处理)
                    //构造一个Collection，虽然Feign官方会将Iterable类型都进行分割，但我们只能处理Collection及其子类（仅限jdk自带的），并不能处理其他Iterable实现类，所以如果使用了其他Iterable自定义类，这里会报错，没法处理
                    Collection<Object> target = createCollection(parameter, vals, isCollectionSplitted);

                    return target;
                } else {//值只有一个，且没有分割Collection，则直接反序列化
                    //使用json反序列化。对于String、int等基本类型也能很好地支持
                    return JSON.parseObject(vals[0], parameterType);
                }
            }
        }catch(IllegalArgumentException e1){
            throw e1;
        }
        catch(Exception e){
            throw genIllegalArgumentException(isCollectionSplitted, parameter, vals, e);
        }
    }

    private Collection<Object> createCollection(MethodParameter parameter, String[] vals, boolean isCollectionSplitted){
        Class clazz = parameter.getParameterType();
        Type parameterType = parameter.getGenericParameterType();

        Type actualType;
        if (ParameterizedType.class.isAssignableFrom(parameterType.getClass())) {
            actualType = ((ParameterizedType) parameterType).getActualTypeArguments()[0];
        } else {
            //can not happen, but still throw Exception here.
            throw genIllegalArgumentException(isCollectionSplitted, parameter, vals);
        }

        //构造一个Collection，虽然Feign官方会将Iterable类型都进行分割，但我们只能处理Collection及其子类（仅限jdk自带的），并不能处理其他Iterable实现类，所以如果使用了其他Iterable自定义类，这里会报错，没法处理
        Collection<Object> target = CollectionFactory.createCollection(clazz,
                actualType instanceof Class ? (Class<?>) actualType : null, vals == null ? 0 : vals.length);

        if(!ArrayUtils.isEmpty(vals)) {
            for (String val : vals) {
                target.add(JSON.parseObject(val, actualType));
            }
        }

        return target;
    }

    private boolean isParamIterableType(Class clazz){
        if (Iterable.class.isAssignableFrom(clazz)) {
            return true;
        }
        return false;
    }

    private IllegalArgumentException genIllegalArgumentException(boolean isCollectionSplitted, MethodParameter p, String[] vals, Exception cause){
        String msg = String.format("error invoking mvc method: %s, because cannot convert from StringArray:#%s# to parameter of type:%s (isCollectionSplitted:%s)(param index:%d)", p.getMethod().getDeclaringClass().getSimpleName() + "." + p.getMethod().getName(), Arrays.toString(vals), p.getGenericParameterType().getTypeName(), isCollectionSplitted?"true":"false", p.getParameterIndex());
        if(cause == null){
            return new IllegalArgumentException(msg);
        }else{
            return new IllegalArgumentException(msg, cause);
        }
    }
    private IllegalArgumentException genIllegalArgumentException(boolean isCollectionSplitted, MethodParameter parameter, String[] vals){
        throw genIllegalArgumentException(isCollectionSplitted, parameter, vals, null);
    }

//    public void printList(List<String> list, String[] arr, List<String>[] arr1){
//        System.out.println(list);
//        System.out.println(Arrays.toString(arr));
//        System.out.println(Arrays.toString(arr1));
//    }
//    public static void main(String[] args) throws NoSuchMethodException, InvocationTargetException, IllegalAccessException {
//        AdvancedFeignClientArgumentResolver instance = new AdvancedFeignClientArgumentResolver();
//        Method printList = AdvancedFeignClientArgumentResolver.class.getDeclaredMethod("printList", List.class, String[].class, List[].class);
//        Type[] types = printList.getGenericParameterTypes();
//        Type type = types[0];
//        System.out.println(type.toString());
//        Object obja = JSONObject.parseObject(" [1]", type);
//        System.out.println(obja);
//
//        type = types[1];
//        System.out.println(GenericArrayType.class.isAssignableFrom(type.getClass()));
//        System.out.println(Object[].class.isAssignableFrom((Class<?>) type));
//        System.out.println(((Class<?>) type).isArray());
//        System.out.println(type.toString());
//        Object objb = JSONObject.parseObject(" [1]", type);
//        System.out.println(objb);
//
//        type = types[2];
//        System.out.println(GenericArrayType.class.isAssignableFrom(type.getClass()));
//        System.out.println((printList.getParameterTypes()[2]).isArray());
//        Type actualType = ((ParameterizedType)((GenericArrayType)type).getGenericComponentType()).getRawType();
//        Object obj1 = JSONObject.parseObject(" [1]", actualType);
//        Object obj2 = JSONObject.parseObject(" [1]", actualType);
//        Object objc = Array.newInstance((Class<?>) actualType, 2);
//        Array.set(objc, 0, obj1);
//        Array.set(objc, 1, obj2);
//
//        System.out.println(objc);
//
//        printList.invoke(instance, obja, objb, objc);
//    }
}
