package cn.com.duibaboot.ext.autoconfigure.threadpool;

import cn.com.duibaboot.ext.autoconfigure.core.utils.PropertyResolver;
import cn.com.duibaboot.ext.autoconfigure.threadpool.factory.ThreadPoolFactoryBean;
import cn.com.duibaboot.ext.autoconfigure.threadpool.properties.DuibaThreadPoolConfig;
import cn.com.duibaboot.ext.autoconfigure.threadpool.properties.ThreadPoolProperties;
import com.google.common.collect.Sets;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.config.BeanDefinition;
import org.springframework.beans.factory.config.ConfigurableListableBeanFactory;
import org.springframework.beans.factory.support.BeanDefinitionBuilder;
import org.springframework.beans.factory.support.BeanDefinitionRegistry;
import org.springframework.beans.factory.support.BeanDefinitionRegistryPostProcessor;
import org.springframework.context.EnvironmentAware;
import org.springframework.core.env.AbstractEnvironment;
import org.springframework.core.env.Environment;
import org.springframework.stereotype.Component;

import java.util.Map;
import java.util.Set;

@Component
public class ThreadPoolBeanFactoryPostProcessor implements BeanDefinitionRegistryPostProcessor,EnvironmentAware {

    private Logger logger = LoggerFactory.getLogger(ThreadPoolBeanFactoryPostProcessor.class);

    private BeanDefinitionRegistry beanDefinitionRegistry;

    private Set<String> threadPoolNames = Sets.newHashSet();

    @Override
    public void postProcessBeanDefinitionRegistry(BeanDefinitionRegistry beanDefinitionRegistry) throws BeansException {
        this.beanDefinitionRegistry = beanDefinitionRegistry;
    }

    @Override
    public void postProcessBeanFactory(ConfigurableListableBeanFactory configurableListableBeanFactory) throws BeansException {

        if(threadPoolNames.isEmpty()){
            return;
        }
        for(String threadPoolName:threadPoolNames){

            BeanDefinitionBuilder builder = BeanDefinitionBuilder.genericBeanDefinition(ThreadPoolFactoryBean.class);
            builder.addPropertyValue("threadPoolName",threadPoolName);

            BeanDefinition beanDefinition = builder.getBeanDefinition();

            Boolean isPrimary = StringUtils.equals(threadPoolName, ThreadPoolProperties.DEFAULT_BEAN_NAME);

            beanDefinition.setPrimary(isPrimary);

            String beanName = threadPoolName;
            if(!beanName.endsWith(ThreadPoolProperties.BEAN_NAME_POSTFIX) && !isPrimary){
                beanName += ThreadPoolProperties.BEAN_NAME_POSTFIX;
            }

            beanDefinitionRegistry.registerBeanDefinition(beanName,beanDefinition);
        }
    }

    @Override
    public void setEnvironment(Environment environment) {

        String threadpoolPrefix = "duiba.threadpool";

        PropertyResolver resolver = new PropertyResolver((AbstractEnvironment)environment,threadpoolPrefix);

        DuibaThreadPoolConfig config = new DuibaThreadPoolConfig();
        resolver.bindToConfig(config);

        if(!config.getEnabled()){
            return;
        }

        threadPoolNames.add(ThreadPoolProperties.DEFAULT_BEAN_NAME);

        Map<String,ThreadPoolProperties> extra = config.getExtra();
        threadPoolNames.addAll(extra.keySet());

        if(config.getMaxSizeCount()>=DuibaThreadPoolConfig.THREAD_MAX_SIZE){
            logger.warn("线程总配置数过高","阁下的系统线程池中的线程总数配置过高，可能导致CPU线程上下文切换频繁，影响性能，建议降低配置总数，或者进行系统拆分，以降低线程池数目");
        }

    }

}
