package cn.com.duibaboot.ext.autoconfigure.web.login;

import com.alibaba.fastjson.JSONObject;
import com.google.common.collect.Sets;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.RestController;
import org.springframework.web.method.HandlerMethod;
import org.springframework.web.servlet.handler.HandlerInterceptorAdapter;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.net.URLEncoder;
import java.util.Set;

public class DuibaBootLoginInterceptor extends HandlerInterceptorAdapter {

    private static final Set<String> EXCLUDE_PATHS = Sets.newHashSet();

    static {
        EXCLUDE_PATHS.add("/favicon.ico");
        EXCLUDE_PATHS.add("/monitor/check");
        EXCLUDE_PATHS.add("/swagger-ui.html");
        EXCLUDE_PATHS.add("/swagger-resources");
        EXCLUDE_PATHS.add("/swagger-resources/configuration/ui");
        EXCLUDE_PATHS.add("/v2/api-docs");
        EXCLUDE_PATHS.add("/error");
    }

    @Autowired
    private LoginStateService loginStateService;
    @Autowired
    private DuibaBootLoginProperties duibaBootLoginProperties;

    @Override
    public boolean preHandle(HttpServletRequest request, HttpServletResponse response, Object handler){

        loginStateService.setRequestInThreadLocal(request,response);

        String url = request.getRequestURI();
        if (EXCLUDE_PATHS.contains(url)){
            return true;
        }
        HandlerMethod handlerMethod = (HandlerMethod) handler;
        IgnoreLoginCheck ignoreLoginCheck = handlerMethod.getMethodAnnotation(IgnoreLoginCheck.class);
        if(ignoreLoginCheck!=null){
            return true;
        }
        LoginState loginState = loginStateService.getCurrentLoginState();
        if(loginState==null || StringUtils.isBlank(loginState.getLoginTicket())){
            returnNotLogin(request,response,handlerMethod);
            return false;
        }
        return true;
    }

    @Override
    public void afterCompletion(HttpServletRequest request, HttpServletResponse response, Object handler, Exception ex){
        loginStateService.removeRequestInThreadLocal();
    }

    /**
     * 判断是否ajax请求
     */
    private boolean isAjaxRequest(HandlerMethod handlerMethod){
        ResponseBody responseBodyAnno = handlerMethod.getMethodAnnotation(ResponseBody.class);
        RestController restAnno = handlerMethod.getBeanType().getAnnotation(RestController.class);
        return(responseBodyAnno != null || restAnno != null || handlerMethod.getMethod().getReturnType().equals(Void.TYPE));//ajax 请求
    }

    /**
     * 跳转到登录页时会带上redirect
     */
    private String getLoginRedirectUrl(HttpServletRequest request){
        return duibaBootLoginProperties.getLoginPagePath() + "?redirect=" + getCurrentRedirect(request);
    }

    private void returnNotLogin(HttpServletRequest request, HttpServletResponse response, HandlerMethod handler){
        try{
            if(isAjaxRequest(handler)){
                response.setHeader("Content-Type", "application/json;charset=UTF-8");
                JSONObject model = new JSONObject();
                model.put("success",false);
                model.put("notLogin",true);
                model.put("message","登录失效，请刷新页面");
                response.getWriter().write(model.toJSONString());
            }else{
                response.sendRedirect(getLoginRedirectUrl(request));
            }
        }catch (Exception e){
            throw new LoginRunTimeException(e);
        }
    }

    /**
     * 获取当前的请求链接
     */
    private String getCurrentRedirect(HttpServletRequest request){
        StringBuilder stringBuilder = new StringBuilder();
        String urlPath = request.getRequestURI();
        if(StringUtils.isBlank(urlPath)){
            stringBuilder.append("/");
        }
        String queryString = request.getQueryString();
        if(StringUtils.isNotBlank(queryString)) stringBuilder.append("?").append(queryString);
        try{
            return URLEncoder.encode(stringBuilder.toString(),"utf-8");
        }catch (Exception e){
            throw new LoginRunTimeException(e);
        }
    }

}
