package cn.com.duibaboot.ext.autoconfigure.accesslog;

import cn.com.duiba.boot.perftest.PerfTestUtils;
import cn.com.duiba.boot.utils.NetUtils;
import cn.com.duiba.wolf.perf.timeprofile.RequestTool;
import cn.com.duiba.wolf.utils.SecurityUtils;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.google.common.base.Charsets;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.util.AntPathMatcher;

import javax.servlet.*;
import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.text.SimpleDateFormat;
import java.util.*;
import java.util.concurrent.ConcurrentHashMap;

/**
 * Created by xuhengfei on 16/12/27.
 */
public class AccessLogFilter implements Filter{

    public static final ThreadLocal<Long> AppId=new ThreadLocal<>();
    public static final ThreadLocal<Long> ConsumerId=new ThreadLocal<>();
    public static final ThreadLocal<Map<String,String>> LogCookie=new ThreadLocal<>();
    public static final ThreadLocal<Map<String,Object>> EX=new ThreadLocal<>();
    public static final ThreadLocal<Map<String,String>> OverWrite=new ThreadLocal<>();

    public static final String OW_HOST="host";

    public static final String ModeBlack="black";//黑名单模式
    public static final String ModeWhite="white";//白名单模式


    private String mode=ModeBlack;
    private Set<String> inhosts=new HashSet<>();

    private Set<String> blackPaths=new HashSet<>();
    private Set<String> blackUriPatterns = new HashSet<>();
    private AntPathMatcher pathMatcher = new AntPathMatcher();

    /**
     * 加载配置项
     * @param properties
     */
    protected void loadConfig(Properties properties){
        mode=properties.getProperty("accesslog.mode");
        String hoststring=properties.getProperty("accesslog.hosts");
        if(hoststring!=null){
            List<String> hosts=Arrays.asList(hoststring.trim().split(","));
            if(!CollectionUtils.isEmpty(hosts)){
                inhosts=new HashSet<>(hosts);
            }
        }
        String blackpaths=properties.getProperty("accesslog.blackpaths");
        List<String> paths=Arrays.asList(StringUtils.split(StringUtils.trimToEmpty(blackpaths),","));
        for(String path : paths){
            if(path != null && !path.contains("*")){
                blackPaths.add(path);
            }else if(path != null && path.contains("*")){
                blackUriPatterns.add(path);
            }
        }
    }

    private String getHost(HttpServletRequest req){
        if(OverWrite.get()!=null){
            String host=OverWrite.get().get(OW_HOST);
            if(host!=null){
                return host;
            }
        }

        return req.getHeader("host");
    }

    /**
     * 是否需要进行日志打印
     * @return
     */
    private boolean needLog(HttpServletRequest req){
        String host=getHost(req);
        if(ModeBlack.equals(mode)){
            if(inhosts.contains(host)){
                return false;
            }
        }else if(ModeWhite.equals(mode)){
            if(!inhosts.contains(host)){
                return false;
            }
        }

        String path=clearRequestURI(req.getRequestURI());
        if(blackPaths.contains(path)){
            return false;
        }
        for(String pattern : blackUriPatterns){
            if(this.pathMatcher.match(pattern, path)){
                return false;
            }
        }

        return true;

    }

    /**
     * 添加cookie信息到线程变量,日志优先从线程变量中获取
     * @param key
     * @param value
     */
    public static void putLogCookie(String key,String value){
        Map<String,String> map=LogCookie.get();
        if(map==null){
            map=new HashMap<>();
            LogCookie.set(map);
        }
        map.put(key,value);
    }

    /**
     * 添加额外信息到线程变量,日志从线程变量获取信息进行打压输出
     * @param key
     * @param value
     */
    public static void putExPair(String key,Object value){
        Map<String,Object> ex=EX.get();
        if(ex==null){
            ex=new HashMap<>();
            EX.set(ex);
        }
        ex.put(key,value);
    }
    /**
     * 添加覆盖信息到线程变量,打印日志时优先从此处获取
     * @param key
     * @param value
     */
    public static void putOverWritePair(String key,String value){
        Map<String,String> map=OverWrite.get();
        if(map==null){
            map=new HashMap<>();
            OverWrite.set(map);
        }
        map.put(key,value);
    }

    private static final Logger accessLog = LoggerFactory.getLogger("duiba_access_log");

    private static final Logger log=LoggerFactory.getLogger(AccessLogFilter.class);

    @Override
    public void init(FilterConfig filterConfig) {
        //do nothing
    }

    @Override
    public void doFilter(ServletRequest servletRequest, ServletResponse servletResponse, FilterChain filterChain) throws IOException, ServletException {

        Long startTime= System.currentTimeMillis();
        Map<String,String> collCookie=new HashMap<>();
        try {
            try {
                //在执行filter之前种cookie,避免resp被提前关闭,无法写入cookie
                addCookieIfNeed((HttpServletRequest) servletRequest, (HttpServletResponse) servletResponse, collCookie);
            }catch(Exception e){
                log.error("addCookieIfNeed error");
            }
            filterChain.doFilter(servletRequest, servletResponse);

        }finally {
            try {
                HttpServletRequest req = (HttpServletRequest) servletRequest;
                //判断是否性能压测url
                boolean isPerfUrl = PerfTestUtils.isPerfTestRequest(req);

                if(!isPerfUrl && needLog(req)){
                    processAccessLog(req,(HttpServletResponse)servletResponse,startTime,collCookie);
                }
            }catch (Exception e){
                log.error("AccessLogFilter process error, message="+e.getMessage());
            }

            AppId.remove();
            ConsumerId.remove();
            LogCookie.remove();
            EX.remove();
            OverWrite.remove();
        }
    }

    private void addCookieIfNeed(HttpServletRequest req,HttpServletResponse resp,Map<String,String> collCookie){
        Map<String,String[]> params=req.getParameterMap();
        String host=req.getHeader("host");
        String cookieHost=getCookieHost(host);

        for(String key:params.keySet()){
            if(!key.startsWith("tck_")){
                continue;
            }

            String[] v=key.split("_");
            if(v.length!=3){
                continue;
            }

            String verify = SecurityUtils.encode2StringByMd5(v[1]);
            if(verify.endsWith(v[2])){
                String name=v[1];
                String value=req.getParameter(key);
                //验证通过
                String cookiekey="_coll_"+name;
                collCookie.put(cookiekey,value);
                //写入cookie

                Cookie cookie = new Cookie(cookiekey, value);
                if(cookieHost!=null) {
                    cookie.setDomain(cookieHost);
                }
                cookie.setPath("/");
                try {
                    resp.addCookie(cookie);
                }catch (Exception e){
                    log.error("addCookie error,cookieKey="+cookiekey+",value="+value,e);
                }
            }
        }
    }

    private void processAccessLog(HttpServletRequest req, HttpServletResponse resp,long startTime,Map<String,String> collCookie) throws Exception{
        //只记录get post请求
        if(!isGetOrPost(req)){
            return;
        }

        Map<String, Object> map = new HashMap<>();
        map.put("url_host", getHost(req));
        map.put("url_path", clearRequestURI(req.getRequestURI()));
        map.put("url_query", req.getQueryString());

        map.put("http_method",req.getMethod());
        map.put("rc",resp.getStatus());
        map.put("rt",(System.currentTimeMillis()-startTime));
        map.put("mip", NetUtils.getLocalIp());//记录服务器IP
        putIfNotNull(map, "ex", EX.get());

        //如果url中有callback参数,认为此url是一个JSONP请求,标记为POST类型,不作为PV统计
        String callback=req.getParameter("callback");
        if(!StringUtils.isBlank(callback)){
            map.put("http_method","POST");
        }

        Map<String, Cookie> cookieMap = getCookieMap(req);

        //优先从线程变量中取数据
        Long consumerId=ConsumerId.get();

        Long appId=AppId.get();

        Cookie ac = cookieMap.get("_ac");
        if (ac != null && (consumerId==null || appId==null)){
            //如果还没有取到,尝试从cookie取数据
            try {
                String json = new String(SecurityUtils.decodeBase64(ac.getValue()), Charsets.UTF_8);
                if(json!=null){
                    JSONObject object = JSONObject.parseObject(json);
                    if(appId==null){
                        appId = object.getLong("aid");
                    }
                    if(consumerId==null){
                        consumerId = object.getLong("cid");
                    }
                }
            }catch(Exception e){
                log.warn("_ac decode fail , _ac= "+ac.getValue());
            }
        }

        putIfNotNull(map, "consumer_id", consumerId);

        putIfNotNull(map, "app_id", appId);

        handleCookies(cookieMap, collCookie);

        handleParams(req, resp, collCookie);

        //支持线程变量中取cookie进行日志打印,便于不同系统的一些定制需求,此为cookie的最高优先级获取方法
        handleLogCookie(collCookie);

        putIfNotEmpty(map, "cookie", collCookie);

        putUa(req, map);

        SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
        map.put("time", sdf.format(new Date()));
        String referer = req.getHeader("referer");
        putIfNotNull(map, "referer", referer);
        String ip = RequestTool.getIpAddr(req);
        putIfNotNull(map, "ip", ip);

        accessLog.info(JSON.toJSONString(map));
    }

    private boolean isGetOrPost(HttpServletRequest req){
        return "get".equalsIgnoreCase(req.getMethod()) || "post".equalsIgnoreCase(req.getMethod());
    }

    private void putUa(HttpServletRequest req, Map<String, Object> map){
        String ua = StringUtils.trimToEmpty(req.getHeader("user-agent"));
        if (ua.length() > 500) {
            ua = ua.substring(0, 499);
        }
        if (!StringUtils.isEmpty(ua)) {
            map.put("user_agent", ua);
        }
    }

    private void putIfNotNull(Map<String, Object> map, String key, Object value){
        if(value != null){
            map.put(key, value);
        }
    }

    private void putIfNotEmpty(Map<String, Object> map, String key, Map<String,String> collCookie){
        if(!collCookie.isEmpty()){
            map.put(key, collCookie);
        }
    }

    private Map<String, Cookie> getCookieMap(HttpServletRequest req){
        Cookie[] cookies = req.getCookies();
        Map<String, Cookie> cookieMap = new HashMap<>();
        if(cookies!=null){
            for (Cookie c : cookies) {
                cookieMap.put(c.getName(), c);
            }
        }
        return cookieMap;
    }

    private void handleCookies(Map<String, Cookie> cookieMap,Map<String,String> collCookie){
        for(Map.Entry<String, Cookie> entry : cookieMap.entrySet()){
            String key = entry.getKey();
            Cookie cookie = entry.getValue();
            if(key.startsWith("_coll_")){
                collCookie.put(key,cookie.getValue());
            }
        }
    }

    private void handleParams(HttpServletRequest req, HttpServletResponse resp, Map<String,String> collCookie){
        Map<String,String[]> params=req.getParameterMap();
        String host=req.getHeader("host");
        String cookieHost=getCookieHost(host);

        for(String key:params.keySet()){
            if(!key.startsWith("tck_")){
                continue;
            }
            String[] v=key.split("_");
            if(v.length!=3){
                continue;
            }
            String verify=SecurityUtils.encode2StringByMd5(v[1]);
            if(verify.endsWith(v[2])){
                String name=v[1];
                String value=req.getParameter(key);
                //验证通过
                String cookiekey="_coll_"+name;
                collCookie.put(cookiekey,value);
                //写入cookie

                Cookie cookie = new Cookie(cookiekey, value);
                if(cookieHost != null) {
                    cookie.setDomain(cookieHost);
                }
                cookie.setPath("/");
                resp.addCookie(cookie);
            }
        }

        //特殊处理
        String slotId=req.getParameter("adslotId");
        if(slotId!=null){
            collCookie.put("_coll_slot",slotId);
        }
        String deviceId=req.getParameter("deviceId");
        if(deviceId!=null){
            collCookie.put("_coll_device",deviceId);
        }
    }

    private void handleLogCookie(Map<String,String> collCookie){
        Map<String,String> logCookie=LogCookie.get();
        if(logCookie!=null){
            for(Map.Entry<String, String> entry : logCookie.entrySet()){
                collCookie.put(entry.getKey(),entry.getValue());
            }
        }
    }

    /**
     * requestURI中可能出现连续两个/，比如//activity//index，需要处理下转为一个/
     * @param requestURI
     * @return
     */
    private String clearRequestURI(String requestURI){
        if(StringUtils.isBlank(requestURI)){
            return requestURI;
        }

        return StringUtils.replace(requestURI, "//", "/");
    }

    private Map<String,String> cookieHosts=new ConcurrentHashMap<>();

    /**
     * 根据host查询 通配符cookie地址
     * @param host
     * @return
     */
    private String getCookieHost(String host){
        if(host==null){
            return null;
        }
        if(host.contains(":")){
            host=host.substring(0,host.indexOf(':'));
        }
        String cookieHost=cookieHosts.get(host);
        if(cookieHost!=null){
            return cookieHost;
        }
        String hostLow=host.toLowerCase();
        if(hostLow.endsWith(".duiba.com.cn") ){
            cookieHost="duiba.com.cn";
        }else if(hostLow.endsWith(".dui88.com")){
            cookieHost="dui88.com";
        }else if(hostLow.endsWith(".duibar.com")){
            cookieHost="duibar.com";
        }else if(hostLow.endsWith(".tuia.cn")) {
            cookieHost = "tuia.cn";
        }

        if(cookieHost!=null){
            cookieHosts.put(host,cookieHost);
        }
        return cookieHost;
    }

    @Override
    public void destroy() {
        //do nothing
    }

}
