package cn.com.duibaboot.ext.autoconfigure.cat;

import cn.com.duiba.boot.cat.CatWithArgs;
import cn.com.duiba.boot.exception.BizException;
import cn.com.duiba.boot.utils.RequestUtils;
import cn.com.duibaboot.ext.autoconfigure.cloud.netflix.feign.CustomRequestInterceptor;
import cn.com.duibaboot.ext.autoconfigure.core.Constants;
import cn.com.duibaboot.ext.autoconfigure.core.rpc.RpcContext;
import cn.com.duibaboot.ext.autoconfigure.core.utils.CatUtils;
import cn.com.duiba.catmonitor.constants.CatConstants;
import cn.com.duiba.boot.netflix.feign.AdvancedFeignClient;
import com.dianping.cat.Cat;
import com.dianping.cat.message.Event;
import com.dianping.cat.message.Message;
import com.dianping.cat.message.Transaction;
import com.dianping.cat.message.internal.AbstractMessage;
import com.dianping.cat.message.internal.DefaultTransaction;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnWebApplication;
import org.springframework.boot.autoconfigure.web.BasicErrorController;
import org.springframework.cloud.netflix.feign.FeignClient;
import org.springframework.context.annotation.Configuration;
import org.springframework.util.ReflectionUtils;
import org.springframework.web.method.HandlerMethod;
import org.springframework.web.servlet.DispatcherServlet;
import org.springframework.web.servlet.handler.HandlerInterceptorAdapter;

import javax.servlet.Servlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.lang.reflect.Method;
import java.util.*;

/**
 * cat rpc interceptor,拦截来自FeignClient的Http Rest Rpc调用并加入cat监控
 */
@Configuration
@ConditionalOnClass({Servlet.class, FeignClient.class})
@ConditionalOnWebApplication
public class CatRpcHandlerInterceptor extends HandlerInterceptorAdapter {
    private static final Logger logger = LoggerFactory.getLogger(CatRpcHandlerInterceptor.class);

    @Value("${spring.application.name}")
    private String springApplicationName;

    private static final String RPC_CAT_CONTEXT_KEY = "RpcCatContext";

    @Override
    public boolean preHandle(HttpServletRequest request, HttpServletResponse response, Object handler) throws Exception {
        if(!(handler instanceof HandlerMethod) || (RequestUtils.getRequestPath(request).equals("/error") && request.getAttribute("javax.servlet.forward.request_uri")!=null)){
            return true;
        }

        //只处理RPC请求（来自FeignClient的rest rpc调用）,如果请求头中含有特殊字段，才认为是rpc请求
        if(!"true".equals(request.getHeader(CustomRequestInterceptor.X_RPC))){
            return true;
        }

        HandlerMethod handlerMethod = (HandlerMethod)handler;
        Method method = handlerMethod.getMethod();
        Class handlerClass = method.getDeclaringClass();
        if(handlerClass.equals(BasicErrorController.class)){
            return true;
        }
        Class feignClientClass = null;
        //找到接口中注解了@FeignClient或@AdvancedFeignClient的接口
        for(Class clazz : handlerClass.getInterfaces()){
            if(clazz.isAnnotationPresent(AdvancedFeignClient.class) || clazz.isAnnotationPresent(FeignClient.class)){
                feignClientClass = clazz;
                break;
            }
        }

        if(feignClientClass == null){
            logger.warn("[NOTIFYME]can not find any interface of class:[{}] which annotated with @FeignClient or @AdvancedFeignClient, will not report to cat", handlerClass.getName());
            return true;
        }
        method = ReflectionUtils.findMethod(feignClientClass, method.getName(), method.getParameterTypes());

        if(method == null){
            logger.warn("[NOTIFYME]feign method is null");
        }

        RpcContext.getContext().setMethod(method);

        if(!CatUtils.isCatEnabled()){
            return true;
        }

        initRpcContext(request, method);
        String loggerName = feignClientClass.getSimpleName()+"."+(method==null?"undefined":method.getName());
        String type = CatConstants.CROSS_SERVER;
        Transaction transaction = Cat.newTransaction(type,loggerName);
        RpcCatContext context = initContext();
        context.setCurrentTransaction(transaction);
        context.setMethod(method);

        createProviderCross(request, transaction);
        Cat.logRemoteCallServer(context);
        setAttachment(context);

        request.setAttribute(RPC_CAT_CONTEXT_KEY, context);

        return true;
    }

    @Override
    public void afterCompletion(HttpServletRequest request, HttpServletResponse response, Object handler, Exception ex) throws Exception {
        // 如果设置了errorView，这里会拦截不到异常, RPC服务端不应该设置errorView
        RpcCatContext context = (RpcCatContext)request.getAttribute(RPC_CAT_CONTEXT_KEY);
        if(context == null){
            return;
        }
        Transaction transaction = context.getCurrentTransaction();
        Method method = context.getMethod();

        if(method != null && method.isAnnotationPresent(CatWithArgs.class)){
            //注解了CatWithArgs表示cat中的监控名需要附加参数，详情见CatWithArgs类的注释
            DefaultTransaction dt = (DefaultTransaction)transaction;
            dt.setName(dt.getName() + resolveArgs(request, method));
        }

        if(ex == null){
            Exception ex1 = (Exception) request.getAttribute(DispatcherServlet.EXCEPTION_ATTRIBUTE);
            if(ex1 != null && ex1 instanceof BizException){
                ex = ex1;
            }
        }

        if(ex != null) {
            if(ex instanceof BizException){//BizException 只记录简略信息到cat

                //记录BizException
                Event bizExceptionEvent = Cat.newEvent("BizException", String.format("BizException(message:%s,code:%s)(this exception will not trigger hystrix)",ex.getMessage(),((BizException) ex).getCode()));
                completeEvent(bizExceptionEvent);
                transaction.addChild(bizExceptionEvent);

                transaction.setStatus(Message.SUCCESS);
            }else {
                Cat.logError(ex);
                transaction.setStatus(ex);
            }
        }else{
            transaction.setStatus(Message.SUCCESS);
        }
//        transaction.complete();
        CatUtils.completeTransaction(transaction);

        RpcContext.removeContext();
    }

    private String resolveArgs(HttpServletRequest request, Method method){
        List<Object> argList = (List<Object>)request.getAttribute(Constants.HTTP_REQUEST_ATTRIBUTE_RPC_ARGS);
        if(argList == null || argList.isEmpty()){
            return "";
        }

        CatWithArgs catWithArgs = method.getAnnotation(CatWithArgs.class);
        int[] argIndexes = catWithArgs.argIndexes();
        if(argIndexes == null || argIndexes.length == 0){
            return "";
        }

        StringBuilder sb = new StringBuilder();
        for(int argIdx : argIndexes){
            if(argIdx >= 0 && argList.size() > argIdx) {
                Object obj = argList.get(argIdx);
                sb.append(obj == null ? "" : obj.toString()).append(",");
            }
        }

        if(sb.length() > 0){
            sb.deleteCharAt(sb.length() - 1);
            sb.insert(0, "(");
            sb.append(")");
        }

        return sb.toString();
    }

    private void initRpcContext(HttpServletRequest req, Method method){
        String catRootId = req.getHeader(Cat.Context.ROOT);
        String catParentId = req.getHeader(Cat.Context.PARENT);
        String catChildId = req.getHeader(Cat.Context.CHILD);
        String catRpcClientName = req.getHeader(Constants.X_RPC_CLIENT);
        RpcContext.getContext().setAttachment(Cat.Context.ROOT, catRootId);
        RpcContext.getContext().setAttachment(Cat.Context.PARENT, catParentId);
        RpcContext.getContext().setAttachment(Cat.Context.CHILD, catChildId);
        RpcContext.getContext().setAttachment(Constants.X_RPC_CLIENT, catRpcClientName);
    }

    static class RpcCatContext implements Cat.Context{

        private Transaction currentTransaction;

        private Method method;

        private Map<String,String> properties = new HashMap<>();

        @Override
        public void addProperty(String key, String value) {
            properties.put(key,value);
        }

        @Override
        public String getProperty(String key) {
            return properties.get(key);
        }

        public Transaction getCurrentTransaction() {
            return currentTransaction;
        }

        public void setCurrentTransaction(Transaction currentTransaction) {
            this.currentTransaction = currentTransaction;
        }

        public Method getMethod() {
            return method;
        }

        public void setMethod(Method method) {
            this.method = method;
        }
    }

    private void setAttachment(Cat.Context context){
        RpcContext.getContext().setAttachment(Cat.Context.ROOT,context.getProperty(Cat.Context.ROOT));
        RpcContext.getContext().setAttachment(Cat.Context.CHILD,context.getProperty(Cat.Context.CHILD));
        RpcContext.getContext().setAttachment(Cat.Context.PARENT,context.getProperty(Cat.Context.PARENT));
    }

    private RpcCatContext initContext(){
        RpcCatContext context = new RpcCatContext();
        Map<String,String> attachments = RpcContext.getContext().getAttachments();
        if(attachments!=null&&attachments.size()>0){
            for(Map.Entry<String,String> entry:attachments.entrySet()){
                if(Cat.Context.CHILD.equals(entry.getKey())||Cat.Context.ROOT.equals(entry.getKey())||Cat.Context.PARENT.equals(entry.getKey())){
                    context.addProperty(entry.getKey(),entry.getValue());
                }
            }
        }
        return context;
    }

    private void completeEvent(Event event){
        AbstractMessage message = (AbstractMessage) event;
        message.setStatus(Event.SUCCESS);
        message.setCompleted(true);
    }

    private void createProviderCross(HttpServletRequest request, Transaction transaction){
        String consumerAppName = RpcContext.getContext().getAttachment(Constants.X_RPC_CLIENT);

        //记录客户端信息
        Event crossServerEvent = Cat.newEvent(CatConstants.PROVIDER_CALL_SERVER, consumerAppName + "[" + request.getRemoteAddr() + ":" + request.getRemotePort() + "]");
        completeEvent(crossServerEvent);
        transaction.addChild(crossServerEvent);

        //记录server端信息
        Event serverEvent = Cat.newEvent("PigeonService.server",springApplicationName + "[" + request.getLocalAddr() + ":" + request.getLocalPort() + "]");
        completeEvent(serverEvent);
        transaction.addChild(serverEvent);
    }

}
