package cn.com.duibaboot.ext.autoconfigure.cloud.netflix.eureka;

import cn.com.duiba.boot.event.MainContextRefreshedEvent;
import cn.com.duiba.boot.utils.AopTargetUtils;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.google.common.base.Joiner;
import com.netflix.discovery.DiscoveryClient;
import com.netflix.discovery.EurekaClient;
import com.netflix.loadbalancer.DynamicServerListLoadBalancer;
import com.netflix.loadbalancer.ILoadBalancer;
import org.apache.commons.lang3.RandomUtils;
import org.apache.http.client.entity.UrlEncodedFormEntity;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.message.BasicNameValuePair;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.cloud.netflix.feign.CustomFeignClientsRegistrar;
import org.springframework.cloud.netflix.ribbon.SpringClientFactory;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.event.EventListener;
import org.springframework.util.ReflectionUtils;

import javax.annotation.PreDestroy;
import javax.annotation.Resource;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.lang.reflect.Method;
import java.util.*;

/**
 *
 * 类的作用：目前某个服务[注册/取消注册]到eurekaServer时，依赖方需要等30秒至1分钟才能知晓，这个类用于与eurekaServer保持长连接，并及时得到依赖变更，从而尽快更新服务列表。
 *
 * <br/>
 *
 * 让eureka客户端使用sse技术连接到eurekaServer上，当eurekaServer检测到服务器注册、取消注册等事件时，通过sse主动通知客户端，随后客户端刷新服务器列表。
 * 本地刷新的动作分为两步：1.让DiscoveryClient主动获取一次服务器列表，2.对于每个应用（比如stock-service），调用对应的DynamicServerListLoadBalancer的updateListOfServers方法，触发更新服务器列表
 * （这里有个地方要注意，由于eurekaServer之间复制信息是异步的，本地可能拿不到刚更新的服务器列表，后续需要观察,如果不行，需要改进,最好是到与sse同个服务器去取）
 */
@Configuration
@ConditionalOnClass(EurekaClient.class)
public class EurekaClientPushAutoConfiguration {

    /**
     * sse事件内容的前缀
     */
    private static final String SSE_PREFIX = "data:";
    /**
     * eureka命令，表示需要客户端刷新某些应用的列表。
     */
    public static final String EUREKA_COMMAND_REFRESH = "refresh";

    private static final Logger logger = LoggerFactory.getLogger(EurekaClientPushAutoConfiguration.class);

    @Resource(name="eurekaClient")
    private EurekaClient eurekaClient;

    @Value("${spring.application.name}")
    private String currentAppName;

    @Resource
    private SpringClientFactory ribbonSpringClientFactory;

    @Resource
    private CloseableHttpClient httpClient;

    private String feignAppNames;

    private int continuousIoExceptionTimes;

    private Thread eurekaClientPushThread = new Thread("eurekaClientPushThread"){
        @Override
        public void run(){
            while(true) {
                try {
                    boolean isContinue = connectEurekaServerSse();
                    if(!isContinue){
                        break;
                    }
                } catch (InterruptedException e) {
                    Thread.currentThread().interrupt();
                } catch (Exception e){
                    logger.error("处理eureka sse失败", e);
                    try {
                        Thread.sleep(10000);
                    } catch (InterruptedException e1) {
                        Thread.currentThread().interrupt();
                    }
                }

                if(Thread.currentThread().isInterrupted()){
                    break;
                }
            }
        }
    };

    @EventListener(MainContextRefreshedEvent.class)
    public void onMainContextRefreshed() {
        //拿到需要用到的所有应用名
        Set<String> feignAppNameList = CustomFeignClientsRegistrar.getEnabledFeignClientNames();
        if(!feignAppNameList.isEmpty()) {
            feignAppNames = Joiner.on(",").join(feignAppNameList);

            eurekaClientPushThread.start();
        }
    }

    @PreDestroy
    public void destroy(){
        eurekaClientPushThread.interrupt();
    }

    /**
     * 与eurekaServer建立长连接并监听感兴趣的服务上下线事件。
     *
     * @return
     * @throws Exception
     */
    private boolean connectEurekaServerSse() throws Exception {
        String oneEurekaServerUrl = getRandomEurekaServerUrl();
        if(oneEurekaServerUrl == null){
            logger.info("eureka server urls is not exists, will not push eureka's registry");
            return false;
        }

        oneEurekaServerUrl = oneEurekaServerUrl.substring(0, oneEurekaServerUrl.lastIndexOf("/eureka")) + "/sse/eureka";

        HttpPost req = new HttpPost(oneEurekaServerUrl);
        UrlEncodedFormEntity entity = new UrlEncodedFormEntity(
                Arrays.asList(new BasicNameValuePair("appName", currentAppName),new BasicNameValuePair("dependentAppNames", feignAppNames)));
        req.setEntity(entity);

        try(CloseableHttpResponse resp = httpClient.execute(req)) {
            if(resp.getEntity() != null
                    && resp.getEntity().getContentType() != null
                    && resp.getEntity().getContentType().getValue() != null) {
                if(resp.getStatusLine().getStatusCode()== 404 || !resp.getEntity().getContentType().getValue().startsWith("text/event-stream")){
                    logger.info("eurekaServer 不支持SSE push");
                    Thread.sleep(60000);
                    return true;
                }
            }
            ByteArrayOutputStream buffer = new ByteArrayOutputStream(1024);
            byte[] bs = new byte[1024];
            while(true) {
                int length = resp.getEntity().getContent().read(bs);
                if (length == -1) {//end of stream
                    break;
                } else if(length > 0) {
                    buffer.write(bs, 0, length);
                    buffer = consumeBuffer(buffer);
                }
            }

            continuousIoExceptionTimes = 0;
        } catch (IOException e) {
            logger.info("连接eurekaServer失败，将会重试:{}", e.getMessage());
            //连续发生5次IOException，则sleep 10秒，以防止死循环耗尽cpu（所有eureka服务都挂掉时会死循环）
            if(continuousIoExceptionTimes++ >= 5){
                Thread.sleep(10000);
            }
        }
        return true;
    }

    private ByteArrayOutputStream consumeBuffer(ByteArrayOutputStream buffer) throws Exception {
        List<String> sseDataLines = getSseDateLines(buffer);

        Set<String> appNamesToRefresh = new HashSet<>();
        for(String sseDataLine : sseDataLines){
            if(sseDataLine.startsWith(SSE_PREFIX)){
                String jsonData = sseDataLine.substring(SSE_PREFIX.length(), sseDataLine.length()-2);//去掉尾部的\n\n
                JSONObject jsonObject = JSON.parseObject(jsonData);
                if(EUREKA_COMMAND_REFRESH.equals(jsonObject.getString("command"))){
                    //appNames 这个字段肯定是全部大写的
                    String appNames = jsonObject.getString("appNames");
                    appNamesToRefresh.addAll(Arrays.asList(appNames.split(",")));
                }
            }else{
                logger.warn("[NOTIFYME]invalid data,ignore it:{}", sseDataLine);
            }
        }
        if(!appNamesToRefresh.isEmpty()) {
            logger.info("notifyRefreshEureka: {}", appNamesToRefresh);
            notifyRefreshEureka(appNamesToRefresh);
        }

        return buffer;
    }

    private List<String> getSseDateLines(ByteArrayOutputStream buffer){
        List<String> sseDataLines = new ArrayList<>();
        boolean isAllDataLinesFetched = false;
        while(!isAllDataLinesFetched && buffer.size() > 1) {
            //这里要考虑多个 data:*\n\n 行出现的情况
            byte[] allBs = buffer.toByteArray();
            for (int i = 0; i < allBs.length - 1; i++) {
                if (allBs[i] == '\n' && i < allBs.length - 1 && allBs[i + 1] == '\n') {
                    String sseDataLine = new String(allBs, 0, i + 2);
                    sseDataLines.add(sseDataLine);

                    buffer.reset();
                    buffer.write(allBs, i + 2, allBs.length - (i + 2));
                    break;
                } else if( i >= allBs.length - 2){//遍历到末尾仍然没有发现匹配的\n\n则直接跳出循环
                    isAllDataLinesFetched = true;
                    break;
                }
            }
        }

        return sseDataLines;
    }

    /**
     * 让eureka客户端使用sse技术连接到eurekaServer上，当eurekaServer检测到服务器注册、取消注册等事件时，通过sse主动通知客户端，随后客户端刷新服务器列表。
     */
    private void notifyRefreshEureka(Set<String> appNamesToRefresh) throws Exception {
        //调用discoveryClient.refreshRegistry方法来触发刷新本地的服务器列表（会获取注册到eureka上的所有服务器）。
        Method method = ReflectionUtils.findMethod(DiscoveryClient.class, "refreshRegistry");
        method.setAccessible(true);
//        if(AopUtils.isJdkDynamicProxy(eurekaClient))
        //eurekaClient可能是refreshScope的，target可能会变，所以需要每次重新获取
        DiscoveryClient discoveryClient = AopTargetUtils.getTarget(eurekaClient);
        ReflectionUtils.invokeMethod(method, discoveryClient);

        //对于需要刷新instances的应用（比如stock-service），调用对应的DynamicServerListLoadBalancer的updateListOfServers方法，触发更新服务器列表
        for(String serviceId : CustomFeignClientsRegistrar.getEnabledFeignClientNames()) {
            if(appNamesToRefresh.contains(serviceId.toUpperCase())) {
                //从ribbon子容器中取得ILoadBalancer实例
                ILoadBalancer loadBalancer = ribbonSpringClientFactory.getInstance(serviceId, ILoadBalancer.class);
                if (loadBalancer instanceof DynamicServerListLoadBalancer) {
                    ((DynamicServerListLoadBalancer) loadBalancer).updateListOfServers();
                }
            }
        }
    }

    /**
     * 随机获取一个eurekaServer的地址
     * @return
     */
    private String getRandomEurekaServerUrl(){
        List<String> eurekaServerUrls = eurekaClient.getEurekaClientConfig().getEurekaServerServiceUrls("defaultZone");
        if(eurekaServerUrls == null || eurekaServerUrls.isEmpty()){
            return null;
        }
        return eurekaServerUrls.get(RandomUtils.nextInt(0, eurekaServerUrls.size()));
    }

}
