package cn.com.duibaboot.ext.autoconfigure.httpclient.ssre;

import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.config.BeanPostProcessor;

import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Set;

public class SsreBeanPostProcessor implements BeanPostProcessor {

    private Logger logger = LoggerFactory.getLogger(SsreBeanPostProcessor.class);

    private final Set<String> whitePackages = Sets.newLinkedHashSet();
    private final Set<ClientWrapper> wrapperClassSet = Sets.newLinkedHashSet();
    private final Class[] acceptableClasses;
    {
        whitePackages.add("cn.com.duibaboot");
        whitePackages.add("org.springframework");
        whitePackages.add("com.netflix");
        whitePackages.add("org");
        whitePackages.add("io");

        wrapperClassSet.add(new SsreHttpClientWrapper());
        wrapperClassSet.add(new SsreHttpAsyncClientWrapper());
        wrapperClassSet.add(new SsreRestTemplateWrapper());
        wrapperClassSet.add(new SsreAsyncRestTemplateWrapper());

        List<Class> acceptableClassesList = new ArrayList<>();
        for(ClientWrapper w : wrapperClassSet) {
            acceptableClassesList.add(w.acceptableClass());
        }
        acceptableClasses = acceptableClassesList.toArray(new Class[acceptableClassesList.size()]);
    }

    @Override
    public Object postProcessBeforeInitialization(Object bean, String beanName) throws BeansException {
        return bean;
    }

    @Override
    public Object postProcessAfterInitialization(Object bean, String beanName) throws BeansException {
        Class clazz = bean.getClass();
        if(!checkPackage(clazz.getPackage())){
            return bean;
        }
        List<Field> fields = getFieldList(clazz);
        for (Field field:fields){
            if (field.isAnnotationPresent(CanAccessInsideNetwork.class)){
                continue;
            }
            Class fieldType = field.getType();
            if(!isAcceptable(fieldType)){
                continue;
            }
            try {
                field.setAccessible(true);
                Object obj = field.get(bean);
                for(ClientWrapper wrapper:wrapperClassSet){
                    if(wrapper.trySetClient(obj)){
                        field.set(bean,wrapper);
                    }
                }
            }catch (Exception e) {
                logger.error("Client代理失败", e);
            }
        }
        return bean;
    }

    private boolean isAcceptable(Class clazz){
        if(clazz.isPrimitive()){
            return false;
        }
        for(Class c : acceptableClasses){
            if(c.isAssignableFrom(clazz)){
                return true;
            }
        }
        return false;
    }

    public void setWhitePackage(Set<String> whitePackage) {
        this.whitePackages.addAll(whitePackage);
    }

    private List<Field> getFieldList(Class clazz){
        if(null == clazz){
            return Collections.emptyList();
        }
        Field[] fields = clazz.getDeclaredFields();
        List<Field> fieldList = Lists.newArrayList(fields);
        Class superClass = clazz.getSuperclass();
        if(superClass.equals(Object.class)){
            return fieldList;
        }
        fieldList.addAll(getFieldList(superClass));
        return fieldList;
    }

    private boolean checkPackage(Package pakeage){
        if(pakeage==null){
            return false;
        }
        String packageName = pakeage.getName();
        for(String whitePackage:whitePackages){
            if(StringUtils.startsWith(packageName,whitePackage)){
                return false;
            }
        }
        return true;
    }

}
