package cn.com.duibaboot.ext.autoconfigure.cloud.netflix.eureka;

import cn.com.duibaboot.ext.autoconfigure.DuibaBootVersion;
import cn.com.duibaboot.ext.autoconfigure.core.utils.SpringBootUtils;
import com.netflix.appinfo.ApplicationInfoManager;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.boot.SpringBootVersion;
import org.springframework.boot.autoconfigure.AutoConfigureAfter;
import org.springframework.boot.autoconfigure.condition.ConditionalOnBean;
import org.springframework.boot.autoconfigure.condition.SearchStrategy;
import org.springframework.cloud.context.config.annotation.RefreshScope;
import org.springframework.cloud.context.scope.refresh.RefreshScopeRefreshedEvent;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.event.EventListener;
import org.springframework.core.Ordered;
import org.springframework.core.annotation.Order;
import org.springframework.core.env.Environment;

import javax.annotation.PostConstruct;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import static org.springframework.context.annotation.ScopedProxyMode.DEFAULT;

/**
 * 注册一些额外的元数据到Eureka（Consul）中，比如服务器启动时间。
 */
@Configuration
@AutoConfigureAfter(name = "org.springframework.cloud.netflix.eureka.EurekaClientAutoConfiguration")
@ConditionalOnBean(type={"com.netflix.appinfo.ApplicationInfoManager"}, search = SearchStrategy.CURRENT)
public class DiscoveryMetadataAutoConfiguration {
    //服务器启动时间
    private static long serverStartUpTime;
    //metadata中的key
    public static final String SERVER_START_UP_TIME_KEY = "serverStartUpTime";
    public static final String WEIGHT_KEY = "weight";
    //是否运行在spring jar in jar 模式下。
    public static final String RUN_IN_SINGLE_JAR_MODE = "runInSingleJarMode";
    public static final String DUIBA_BOOT_VERSION = "duibaBootVersion";
    public static final String SPRING_BOOT_VERSION = "springBootVersion";
    public static final String CONFIG_VERSION = "configVersion";
    //是否运行在Docker容器中(如果这个key设置在系统环境变量中，则key为 RUN_IN_DOCKER)
    public static final String RUN_IN_DOCKER = "run.in.docker";
    //压测场景id，如果eureka注册信息中包含此字段，表示这个容器只用于线上压测，正常流量不应该调度到这个容器，而且不同场景id之间流量也要隔离(如果这个key设置在系统环境变量中，则key为 DUIBA_PERF_SCENE_ID)
    //详情见此文档：  http://cf.dui88.com/pages/viewpage.action?pageId=10869508
    public static final String DUIBA_PERF_SCENE_ID = "duiba.perf.scene.id";

    //applicationInfoManager这个bean有可能是refreshScope的，refresh之后bean实例会变化(实际上拿到的是一个不变的Proxy，实际的会变的对象封装在内部)，这里能确保每次获取这个bean都拿到最新的
    @Autowired
    private ApplicationInfoManager applicationInfoManager;

    @Autowired
    private Environment environment;

    @Value("${"+RUN_IN_DOCKER+":false}")
    private boolean runInDocker;
    @Value("${"+DUIBA_PERF_SCENE_ID+":}")
    private String duibaPerfSceneId;

    /**
     * 获取服务启动时间，确保只生成一次。
     * @return
     */
    private synchronized long getServerStartUpTime(){
        if(serverStartUpTime == 0){
            serverStartUpTime = System.currentTimeMillis();
        }
        return serverStartUpTime;
    }

    //applicationInfoManager这个bean有可能是refreshScope的,所以需要监听refresh事件，监听到变化后重新设置metadata信息。
    //@ConditionalOnClass(RefreshScopeRefreshedEvent.class)?
    @EventListener(value = {RefreshScopeRefreshedEvent.class})
    @Order(Ordered.HIGHEST_PRECEDENCE)
    public void onRefresh(RefreshScopeRefreshedEvent event){
        registerAppMetadataCommon();
    }

    @PostConstruct
    public void init(){
        registerAppMetadataCommon();
    }

    @Autowired(required = false)
    private List<DiscoveryMetadataRegister> discoveryMetadataRegisters;

    private void registerAppMetadataCommon(){
        long serverStartUpTime = getServerStartUpTime();

        if(applicationInfoManager != null) {
            Map<String, String> metadata = new HashMap<>();
            //metadata中放入服务器启动时间
            metadata.put(SERVER_START_UP_TIME_KEY, String.valueOf(serverStartUpTime));

            //metadata中放入服务器权重,默认100
            //TODO 这里有点小问题，如果有人改了权重，refresh之后权重会被重置回100,后期改掉,把100作为变量放在某处，允许改变
            metadata.put(WEIGHT_KEY, String.valueOf(100));

            metadata.put(RUN_IN_SINGLE_JAR_MODE, SpringBootUtils.isJarInJarMode() ? "true" : "false");
            metadata.put(SPRING_BOOT_VERSION, StringUtils.trimToEmpty(SpringBootVersion.getVersion()));
            metadata.put(DUIBA_BOOT_VERSION, StringUtils.trimToEmpty(DuibaBootVersion.getVersion()));
            metadata.put(RUN_IN_DOCKER, runInDocker ? "true" : "false");
            if(!StringUtils.isBlank(duibaPerfSceneId)) {
                metadata.put(DUIBA_PERF_SCENE_ID, duibaPerfSceneId);
            }
            metadata.put(CONFIG_VERSION, StringUtils.trimToEmpty(environment.getProperty("config.client.version")));

            if(discoveryMetadataRegisters != null) {
                for (DiscoveryMetadataRegister register : discoveryMetadataRegisters) {
                    register.registerMetadata(metadata);
                }
            }

            applicationInfoManager.registerAppMetadata(metadata);
        }
    }

}
