package cn.com.duibaboot.ext.autoconfigure.logger.filter;

import ch.qos.logback.classic.spi.ILoggingEvent;
import ch.qos.logback.core.filter.Filter;
import ch.qos.logback.core.spi.FilterReply;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.boot.context.properties.ConfigurationProperties;

/**
 * 日志打印单条大小限制(由于开发对日志打印没有明确的概念，导致单条日志内容过大，导致flume采集出现报错，影响线上业务)
 * 需求描述 （阈值支持可配置）
 *     当单条日志超过16K，出现告警
 *
 *     当单条日志超过32K，日志不打印
 */
@ConfigurationProperties(prefix = "duiba.logger.size-limit")
public class LoggerSizeFilter extends Filter<ILoggingEvent> {

    /**
     * 日志大小限制阈值，单位：字节, 默认值：32768,取值范围 [1024, +∞)，当日志大小超过此阈值时，日志不会打印，并同时打印一条error告警日志，如果频繁出现，会触发日志告警
     */
    private int errorThreshold = 32*1024;
    /**
     * 日志大小限制阈值，单位：字节, 默认值：16384,取值范围 [1024, +∞)，当日志大小超过此阈值时，日志会打印，并同时打印一条warn告警日志，如果频繁出现，会触发日志告警
     */
    private int warnThreshold = 16*1024;

    private static final Logger logger = LoggerFactory.getLogger("LoggerSizeFilter");

    @Override
    public FilterReply decide(ILoggingEvent event) {
        if(event == null
                || event.getThrowableProxy() != null //有异常则不限制。
                || event.getLoggerName().equals(logger.getName())){//不限制LoggerSizeFilter打印的日志，以避免死循环
            return FilterReply.NEUTRAL;
        }
        int loggerSize = event.getFormattedMessage().length();
        //duiba.logger.size-limit.error-threshold=32K
        //duiba.logger.size-limit.warn-threshold=16K
        if(loggerSize > errorThreshold){//日志超大，打印告警日志，并禁止此日志打印
            logger.warn("logsize:{} exceed duiba.logger.size-limit.error-threshold: {}, loggerName:{}", loggerSize, errorThreshold, event.getLoggerName());
            return FilterReply.DENY;
        }else if(loggerSize > warnThreshold){//日志过大，打印告警日志，允许此日志打印
            logger.error("logsize:{} exceed duiba.logger.size-limit.warn-threshold: {}, loggerName:{}", loggerSize, warnThreshold, event.getLoggerName());
            return FilterReply.NEUTRAL;
        }
        return FilterReply.NEUTRAL;
    }


    public int getErrorThreshold() {
        return errorThreshold;
    }

    public void setErrorThreshold(int errorThreshold) {
        if(errorThreshold < 1024 ){
            logger.warn("property `duiba.logger.size-limit.error-threshold={}` is invalid, will use default value{}", errorThreshold, this.errorThreshold);
            return;
        }
        this.errorThreshold = errorThreshold;
    }

    public int getWarnThreshold() {
        return warnThreshold;
    }

    public void setWarnThreshold(int warnThreshold) {
        if(warnThreshold < 1024 ){
            logger.warn("property `duiba.logger.size-limit.warn-threshold={}` is invalid, will use default value{}", warnThreshold, this.warnThreshold);
            return;
        }
        this.warnThreshold = warnThreshold;
    }

}
