package org.springframework.cloud.netflix.feign;

import cn.com.duiba.boot.exception.BizException;
import cn.com.duibaboot.ext.autoconfigure.core.rpc.RpcContext;
import cn.com.duibaboot.ext.autoconfigure.core.utils.CatUtils;
import cn.com.duiba.catmonitor.constants.CatConstants;
import com.alibaba.dubbo.common.Constants;
import com.alibaba.dubbo.common.URL;
import com.dianping.cat.Cat;
import com.dianping.cat.message.Event;
import com.dianping.cat.message.Message;
import com.dianping.cat.message.Transaction;
import com.dianping.cat.message.internal.AbstractMessage;
import com.google.common.annotations.VisibleForTesting;
import com.netflix.hystrix.exception.HystrixBadRequestException;
import org.aopalliance.intercept.MethodInterceptor;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Value;

import java.lang.reflect.InvocationHandler;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.lang.reflect.Proxy;
import java.util.HashMap;
import java.util.Map;


/**
 * 拦截FeignClient的调用加入cat监控功能，以及拦截并注入当前调用的方法
 */
public class CustomFeignClientFactoryBean extends FeignClientFactoryBean {

    @Value("${spring.application.name}")
    private String springApplicationName;

    @Override
    public Object getObject() throws Exception {
        //增加cat监控功能
        Class classType = super.getObjectType();
        Object obj = super.getObject();

        return Proxy.newProxyInstance(classType.getClassLoader(), new Class[]{classType}, new FeignClientInvocationHandler(springApplicationName, obj));
    }


    /**
     * {@link MethodInterceptor}
     */
    @VisibleForTesting
    public static class FeignClientInvocationHandler implements InvocationHandler {

        private String springApplicationName;
        private static final ThreadLocal<Cat.Context> CAT_CONTEXT = new ThreadLocal<>();
        private Object original;

        FeignClientInvocationHandler(String springApplicationName, Object original) {
            this.springApplicationName = springApplicationName;
            this.original = original;
        }

        @Override
        public Object invoke(Object proxy, Method method, Object[] args) throws Throwable {
            String methodName = method.getName();

            if ("equals".equals(methodName)
                    || "hashCode".equals(methodName)
                    || "toString".equals(methodName)
                    || method.isDefault()) {
                return invokeMethodInner(method, original, args);
            }

            RpcContext.getContext().setMethod(method);

            if(!CatUtils.isCatEnabled()){
                return invokeMethodInner(method, original, args);
            }

            String loggerName = method.getDeclaringClass().getSimpleName()+"."+method.getName();
            Transaction transaction = Cat.newTransaction(CatConstants.CROSS_CONSUMER,loggerName);

            try {
                Cat.Context context = getContext();
                //服务端信息让服务端打印，由于hystrix的存在，客户端打印不方便
                //createConsumerCross(url,transaction);

                Cat.logRemoteCallClient(context);
                setAttachment(context, method);

                Object result = invokeMethodInner(method, original, args);

                transaction.setStatus(Message.SUCCESS);
                return result;
            }catch(Throwable e){
                if(!(e instanceof BizException)) {
                    Cat.logError(e);//BizException是正常的业务异常，比如用户不存在，不应该记录到cat.
                    transaction.setStatus(e);
                }else{
                    transaction.setStatus(String.format("BizException(message:%s,code:%s)(this exception will not trigger hystrix)",e.getMessage(),((BizException) e).getCode()));
                }
                throw e;
            }finally {
                CatUtils.completeTransaction(transaction);
                CAT_CONTEXT.remove();
            }
        }

        private Object invokeMethodInner(Method method, Object target, Object[] args) throws Throwable{
            try {
                return method.invoke(target, args);
            } catch (Throwable e) {
                if(e instanceof InvocationTargetException){
                    e = ((InvocationTargetException)e).getTargetException();
                }
                //通常这步拿出来的异常只会有以下2种：HystrixBadRequestException, FeignClientException
                if(e instanceof HystrixBadRequestException && e.getCause() != null && e.getCause() instanceof BizException){
                    e = e.getCause();//如果cause是BizException，则应该把BizException提取出来抛出
                }
                throw e;
            }
        }

        private static class RpcCatContext implements Cat.Context{

            private Map<String,String> properties = new HashMap<String, String>();

            @Override
            public void addProperty(String key, String value) {
                properties.put(key,value);
            }

            @Override
            public String getProperty(String key) {
                return properties.get(key);
            }
        }

        private String getProviderAppName(URL url){
            String appName = url.getParameter(CatConstants.PROVIDER_APPLICATION_NAME);
            if(StringUtils.isEmpty(appName)){
                String interfaceName  = url.getParameter(Constants.INTERFACE_KEY);
                appName = interfaceName.substring(0,interfaceName.lastIndexOf('.'));
            }
            return appName;
        }

        private void setAttachment(Cat.Context context, Method method){
            RpcContext.getContext().setAttachment(Cat.Context.ROOT,context.getProperty(Cat.Context.ROOT));
            RpcContext.getContext().setAttachment(Cat.Context.CHILD,context.getProperty(Cat.Context.CHILD));
            RpcContext.getContext().setAttachment(Cat.Context.PARENT,context.getProperty(Cat.Context.PARENT));
            RpcContext.getContext().setAttachment(cn.com.duibaboot.ext.autoconfigure.core.Constants.X_RPC_CLIENT,springApplicationName);
        }

        private Cat.Context getContext(){
            Cat.Context context = CAT_CONTEXT.get();
            if(context==null){
                context = new RpcCatContext();
                CAT_CONTEXT.set(context);
            }
            return context;
        }

        private void createConsumerCross(URL url,Transaction transaction){
            Event crossAppEvent =  Cat.newEvent(CatConstants.CONSUMER_CALL_APP,getProviderAppName(url));
            Event crossServerEvent = Cat.newEvent(CatConstants.CONSUMER_CALL_SERVER,url.getHost());
            Event crossPortEvent = Cat.newEvent(CatConstants.CONSUMER_CALL_PORT,url.getPort()+"");
            completeEvent(crossAppEvent);
            completeEvent(crossPortEvent);
            completeEvent(crossServerEvent);
            transaction.addChild(crossAppEvent);
            transaction.addChild(crossPortEvent);
            transaction.addChild(crossServerEvent);
        }

        private void completeEvent(Event event){
            AbstractMessage message = (AbstractMessage) event;
            message.setCompleted(true);
            message.setStatus(Event.SUCCESS);
        }

    }

}
