package cn.com.duibaboot.ext.autoconfigure.cloud.netflix.hystrix;

import org.aopalliance.intercept.MethodInterceptor;
import org.aopalliance.intercept.MethodInvocation;
import org.springframework.aop.framework.ProxyFactory;

import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpServletResponseWrapper;
import java.io.IOException;
import java.io.PrintWriter;

/**
 * 这个类用于修复/hystrix.stream的bug：https://github.com/Netflix/Hystrix/issues/1756
 * <br/>
 * 对HttpServletResponse.PrintWriter的方法进行aop,从而进行必要的同步保护,以修复HystrixSampleSseServlet多线程往writer中写导致tomcat出错的问题。
 * <br/>
 * 等官方修复这个bug后可以去掉这个类。
 */
public class ThreadSafeHttpServletResponseWrapper extends HttpServletResponseWrapper {

    private PrintWriter delegateWriter;

    public ThreadSafeHttpServletResponseWrapper(HttpServletResponse response) {
        super(response);
    }

    @Override
    public synchronized PrintWriter getWriter() throws IOException {
        if(delegateWriter == null) {
            PrintWriter writer = super.getWriter();
            ProxyFactory factory = new ProxyFactory();
            factory.setTarget(writer);
            factory.addAdvice(new PrintWriterMethodInterceptor());
            delegateWriter = (PrintWriter) factory.getProxy();
        }

        return delegateWriter;
    }

    /**
     * 对PrintWriter的方法进行aop,从而进行必要的同步保护,以修复HystrixSampleSseServlet多线程往writer中写导致的混乱问题。
     */
    static class PrintWriterMethodInterceptor implements MethodInterceptor {

        private volatile boolean errorOccurred;

        @Override
        public Object invoke(MethodInvocation invocation) throws Throwable {
            String methodName = invocation.getMethod().getName();
            PrintWriter originalWriter = ((PrintWriter)invocation.getThis());
            if(methodName.equals("print") || methodName.equals("write")
                || methodName.equals("println")){
                synchronized (originalWriter){
                    Object obj = invocation.proceed();

                    errorOccurred = originalWriter.checkError();

                    originalWriter.flush();

                    return obj;
                }
            }else if(methodName.equals("checkError")){
                return errorOccurred;
            }else if(methodName.equals("flush")){
                //do nothing
                return null;
            }
            return invocation.proceed();
        }
    }
}
