package cn.com.duibaboot.ext.autoconfigure.flowreplay;

import com.alibaba.fastjson.JSONObject;
import com.google.common.base.Joiner;
import org.apache.commons.lang3.StringUtils;
import org.aspectj.lang.reflect.MethodSignature;

import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

/**
 * 引流回归一些工具方法
 * Created by guoyanfei .
 * 2019-03-14 .
 */
public final class FlowReplayUtils {

    private FlowReplayUtils() {
    }

    /**
     * 判断一个字符串是否是json
     * @param json
     * @return
     */
    public static boolean isJSONValid(String json) {
        try {
            JSONObject.parseObject(json);
        } catch (Exception e0) {
            try {
                JSONObject.parseArray(json);
            } catch (Exception e1) {
                return false;
            }
        }
        return true;
    }

    /**
     * 判断是否是回放专用环境
     * @return
     */
    public static boolean isReplayEnv() {
        String duibaFlowReplayReportId = System.getenv("DUIBA_FLOW_REPLAY_REPORT_ID");
        return StringUtils.isNotBlank(duibaFlowReplayReportId);
    }

    /**
     * 根据方法，解出类似以下格式的字符串
     * RemoteConsumerService.getConsumer(Long)
     * @param method
     * @return
     */
    public static String parseApiNameByMethod(Method method) {
        StringBuilder sb = new StringBuilder();
        appendType(sb, method.getDeclaringClass());
        sb.append(".");
        sb.append(method.getName());
        sb.append("(");
        Class<?>[] parametersTypes = method.getParameterTypes();
        appendTypes(sb, parametersTypes);
        sb.append(")");
        return sb.toString();
    }

    /**
     * 调用的方法是否一样
     * @param spanMethod
     * @param currentMethod
     * @return
     */
    public static boolean isMethodEqual(String spanMethod, Method currentMethod) {
        return spanMethod.equals(currentMethod.getName());
    }

    /**
     * 调用的参数类型是否一样
     * @param spanParameterTypes
     * @param currentParameterTypes
     * @return
     */
    public static boolean isArgumentsTypesEqual(String[] spanParameterTypes, Class<?>[] currentParameterTypes) {
        for (int i = 0; i < currentParameterTypes.length; i++) {
            if (!currentParameterTypes[i].getName().equals(spanParameterTypes[i])) {
                return false;
            }
        }
        return true;
    }

    public static String stringArrayToString(String[] stringArray) {
        if (stringArray == null || stringArray.length == 0) {
            return StringUtils.EMPTY;
        }
        return Joiner.on(",").join(stringArray);
    }

    public static String classArrayToString(Class<?>[] classArray) {
        if (classArray == null || classArray.length == 0) {
            return StringUtils.EMPTY;
        }
        List<String> classNames = new ArrayList<>(classArray.length);
        for (Class<?> c : classArray) {
            classNames.add(c.getName());
        }
        return Joiner.on(",").join(classNames);
    }

    /**
     * 调用的参数值是否一样
     * @param spanParameterValues
     * @param currentParameterValues
     * @return
     */
    public static boolean isArgumentsEqual(Object[] spanParameterValues, Object[] currentParameterValues) {
        for (int i = 0; i < currentParameterValues.length; i++) {
            Object arg = currentParameterValues[i];
            Object spanArg = spanParameterValues[i];
            if (arg instanceof byte[]) {
                if (!Arrays.equals((byte[]) arg, (byte[]) spanArg)) {
                    return false;
                }
            } else {
                if (!arg.equals(spanArg)) {
                    return false;
                }
            }
        }
        return true;
    }

    /**
     * 全局拦截白名单，白名单的不拦截
     * @param allArguments
     * @return
     */
    public static boolean isGlobalWhitelist(Object[] allArguments) {
        if (allArguments != null) {
            for (Object obj : allArguments) {
                if (FlowReplayConstants.CANNOT_DESERIALIZE_CLASSES.contains(obj.getClass().getName())) {
                    return true;
                }
            }
        }
        return false;
    }

    /**
     * guavacache的方法拦截器白名单
     * @param className
     * @return
     */
    public static boolean isGuavaWhitelist(String className) {
        if (FlowReplayConstants.GUAVA_WHITELIST.contains(className)) {
            return true;
        }
        return false;
    }

    /**
     * caffeine cache 的方法拦截器白名单
     * @param className
     * @return
     */
    public static boolean isCaffeineWhitelist(String className) {
        if (FlowReplayConstants.CAFFEINE_WHITELIST.contains(className)) {
            return true;
        }
        return false;
    }

    /**
     * 根据方法签名，解出类似以下格式的字符串
     * RemoteConsumerService.getConsumer(Long)
     * @param methodSignature
     * @return
     */
    public static String parseApiNameByMethodSignature(MethodSignature methodSignature) {
        StringBuilder sb = new StringBuilder();
        appendType(sb, methodSignature.getDeclaringType());
        sb.append(".");
        sb.append(methodSignature.getMethod().getName());
        sb.append("(");
        Class<?>[] parametersTypes = methodSignature.getParameterTypes();
        appendTypes(sb, parametersTypes);
        sb.append(")");
        return sb.toString();
    }

    private static void appendTypes(StringBuilder sb, Class<?>[] types) {
        for (int size = types.length, i = 0; i < size; i++) {
            appendType(sb, types[i]);
            if (i < size - 1) {
                sb.append(",");
            }
        }
    }

    private static void appendType(StringBuilder sb, Class<?> type) {
        if (type.isArray()) {
            appendType(sb, type.getComponentType());
            sb.append("[]");
        } else {
            sb.append(type.getSimpleName());
        }
    }
}
