package cn.com.duibaboot.ext.autoconfigure.flowreplay.replay;

import cn.com.duibaboot.ext.autoconfigure.flowreplay.FlowReplayErrorMsgTypeEnum;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.FlowReplaySpan;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.FlowReplayTrace;
import com.alibaba.ttl.TransmittableThreadLocal;
import com.google.common.base.Joiner;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.LinkedList;
import java.util.List;

/**
 * 回放的单个trace上下文
 * Created by guoyanfei .
 * 2019-02-27 .
 */
@Slf4j
public class ReplayTraceContext implements Serializable {

    private static final long serialVersionUID = 2716578641071393051L;

    private String traceId;

    private FlowReplaySpan mainSpan;

    private LinkedList<FlowReplaySpan> subSpans;

    /**
     * 标记了用例失败
     */
    private boolean errorMarked;

    /**
     * 错误原因
     */
    private FlowReplayErrorMsgTypeEnum errorMsgType;

    /**
     * 错误详情
     */
    private String errorMsgDetail;

    private static class TransmittableThreadLocalHolder {

        private TransmittableThreadLocalHolder() {
        }

        private static final TransmittableThreadLocal<ReplayTraceContext> threadLocal2ReplayTraceContext = new TransmittableThreadLocal<>();
    }

    /**
     * 创建trace
     * @param mainSpan
     */
    public static void create(FlowReplayTrace trace) {
        ReplayTraceContext c = new ReplayTraceContext();
        c.traceId = trace.getTraceId();
        c.mainSpan = trace.getMainSpan();
        c.subSpans = new LinkedList<>(trace.getSubSpans());
        c.errorMarked = false;
        c.errorMsgType = null;
        c.errorMsgDetail = null;
        TransmittableThreadLocalHolder.threadLocal2ReplayTraceContext.set(c);
        traceLog(c);
    }

    private static void traceLog(ReplayTraceContext c) {
        if (log.isDebugEnabled()) {
            List<String> subSpanTypes = new ArrayList<>();
            for (FlowReplaySpan span : c.subSpans) {
                subSpanTypes.add(span.getSpanType().toString());
            }
            log.debug("Replay_TraceId_{}_MainSpanType_{}_SubSpanTypes_{}", c.traceId, c.mainSpan.getSpanType(), Joiner.on("|").join(subSpanTypes));
        }
    }

    /**
     * 标记本次回放请求为异常
     * @param errorMsgType
     * @param expert
     * @param actual
     * @param exceptionMsg
     */
    private static void markError(FlowReplayErrorMsgTypeEnum errorMsgType, String expert, String actual, String exceptionMsg) {
        ReplayTraceContext c = TransmittableThreadLocalHolder.threadLocal2ReplayTraceContext.get();
        if (c == null || c.errorMarked) {   // context未创建 || 已经被标记过
            return;
        }
        c.errorMarked = true;
        c.errorMsgType = errorMsgType;
        // 优先展示异常信息
        if (StringUtils.isNotBlank(exceptionMsg)) {
            c.errorMsgDetail = exceptionMsg;
        } else if (StringUtils.isNotBlank(expert) || StringUtils.isNotBlank(actual)) {
            c.errorMsgDetail = String.format("期望: %s, 实际: %s", expert, actual);
        }
    }

    /**
     * 标记本次回放请求为异常
     * @param errorMsgType
     */
    public static void markError(FlowReplayErrorMsgTypeEnum errorMsgType) {
        markError(errorMsgType, null, null, null);
    }

    /**
     * 标记本次回放请求为异常
     * @param errorMsgType
     * @param expert
     * @param actual
     */
    public static void markError(FlowReplayErrorMsgTypeEnum errorMsgType, String expert, String actual) {
        markError(errorMsgType, expert, actual, null);
    }

    /**
     * 标记本次回放请求为异常
     * @param errorMsgType
     * @param exceptionMsg
     */
    public static void markError(FlowReplayErrorMsgTypeEnum errorMsgType, String exceptionMsg) {
        markError(errorMsgType, null, null, exceptionMsg);
    }

    /**
     * 获取并删除span队列头部的span
     * @return
     */
    public static FlowReplaySpan pollSubSpan() {
        ReplayTraceContext c = TransmittableThreadLocalHolder.threadLocal2ReplayTraceContext.get();
        if (c == null || c.subSpans == null) {
            return null;
        }
        return c.subSpans.poll();
    }

    /**
     * 获取并删除当前回放请求上下文
     * @return
     */
    public static ReplayTraceContext getAndRemove() {
        ReplayTraceContext c = TransmittableThreadLocalHolder.threadLocal2ReplayTraceContext.get();
        if (c == null) {
            return null;
        }
        TransmittableThreadLocalHolder.threadLocal2ReplayTraceContext.remove();
        return c;
    }

    public static ReplayTraceContext get() {
        ReplayTraceContext c = TransmittableThreadLocalHolder.threadLocal2ReplayTraceContext.get();
        if (c == null) {
            return null;
        }
        return c;
    }

    public static String getContextTraceId() {
        ReplayTraceContext c = TransmittableThreadLocalHolder.threadLocal2ReplayTraceContext.get();
        if (c == null) {
            return null;
        }
        return c.traceId;
    }

    public static void remove() {
        TransmittableThreadLocalHolder.threadLocal2ReplayTraceContext.remove();
    }

    /**
     * 是否正在回放(本次请求是否是回放请求)
     * @return
     */
    public static boolean isReplaying() {
        ReplayTraceContext c = TransmittableThreadLocalHolder.threadLocal2ReplayTraceContext.get();
        return c != null;
    }

    public String getTraceId() {
        return traceId;
    }

    public FlowReplaySpan getMainSpan() {
        return mainSpan;
    }

    public LinkedList<FlowReplaySpan> getSubSpans() {
        return subSpans;
    }

    public boolean isErrorMarked() {
        return errorMarked;
    }

    public FlowReplayErrorMsgTypeEnum getErrorMsgType() {
        return errorMsgType;
    }

    public void setTraceId(String traceId) {
        this.traceId = traceId;
    }

    public String getErrorMsgDetail() {
        return errorMsgDetail;
    }
}
