package cn.com.duibaboot.ext.autoconfigure.flowreplay.replay;

import cn.com.duibaboot.ext.autoconfigure.flowreplay.FlowReplayFileComponent;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.FlowReplayTrace;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.replay.event.ReplayStartEvent;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.serializer.Hessian2Serializer;
import lombok.extern.slf4j.Slf4j;
import org.springframework.context.event.EventListener;

import javax.annotation.Resource;
import java.io.File;
import java.io.IOException;
import java.io.RandomAccessFile;

/**
 * 下载并解压录制的用例集文件，然后把用例读取到内存中，供用例消费者消费
 * Created by guoyanfei .
 * 2019-02-26 .
 */
@Slf4j
public class ReplayTraceLoader {

    @Resource
    private FlowReplayFileComponent flowReplayFileComponent;

    /**
     * 监听 "回放开始" 事件，加载用例集文件内容到内存中
     * @param event
     */
    @EventListener(ReplayStartEvent.class)
    public void replayStartEventListener(ReplayStartEvent event) {
        new Thread(new Runnable() {

            @Override
            public void run() {
                ReplayContext context = ReplayContextHolder.getReplayContext();
                if (context == null) {
                    return;
                }
                try {
                    loadUsecaseSet(context);
                } catch (IOException e) {
                    log.error("loadUsecaseSet_error", e);
                } finally {
                    // 用例集加载完成
                    context.tracesLoadFinish();
                }
            }
        }, "DBThread-Replay-LoadUsecaseSet").start();
    }

    /**
     * 加载用例集
     * @throws IOException
     */
    private void loadUsecaseSet(ReplayContext context) throws IOException {
        String localReplayPath = flowReplayFileComponent.download(context.getUsecaseSetUrl());
        File file = new File(localReplayPath);
        try (RandomAccessFile raf = new RandomAccessFile(file, "rw")) {
            raf.skipBytes(8);   // 跳过版本号和用例数
            while (raf.getFilePointer() < raf.length()) {
                int length = raf.readInt();
                byte[] traceBytes = new byte[length];
                for (int i = 0; i < length; i++) {
                    traceBytes[i] = raf.readByte();
                }
                FlowReplayTrace trace = (FlowReplayTrace) Hessian2Serializer.deserialize(traceBytes);
                context.offerTrace(trace);
            }
        }
    }
}
