package cn.com.duibaboot.ext.autoconfigure.flowreplay.replay;

import cn.com.duibaboot.ext.autoconfigure.flowreplay.FlowReplayTrace;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.replay.event.ReplayStartEvent;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.replay.replayer.Replayer;
import lombok.extern.slf4j.Slf4j;
import org.springframework.context.event.EventListener;

import javax.annotation.Resource;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;

/**
 * 多线程消费用例加载器加载到内存中的用例，调用回放器回放对应类型的用例
 * Created by guoyanfei .
 * 2019-02-26 .
 */
@Slf4j
public class ReplayTraceReplayer {

    /**
     * 回放的线程池大小
     */
    private static final int REPLAYER_POOL_SIZE = 10;

    /**
     * 回放器
     */
    @Resource
    private Replayer replayer;

    private ExecutorService es = Executors.newFixedThreadPool(REPLAYER_POOL_SIZE);

    /**
     * 监听"回放开始"事件，多线程启动多个回放器
     * @param event
     * @throws ExecutionException
     * @throws InterruptedException
     */
    @EventListener(ReplayStartEvent.class)
    public void replayStartEventListener(ReplayStartEvent event) {
        new Thread(new Runnable() {
            @Override
            public void run() {
                ReplayContext context = ReplayContextHolder.getReplayContext();
                if (context == null) {
                    return;
                }
                try {
                    List<Future> futures = new ArrayList<>(REPLAYER_POOL_SIZE);
                    for (int i = 0; i < REPLAYER_POOL_SIZE; i++) {
                        futures.add(es.submit(new ReplayerThread(context)));
                    }
                    for (Future it : futures) {
                        try {
                            it.get();
                        } catch (Exception e) {
                            log.error("ReplayerThread_Future_get_error", e);
                        }
                    }
                } finally {
                    // 标记回放操作已经完成，但是结果队列可能还在写文件
                    context.replayersDone();
                }
            }
        }, "DBThread-Replay-Replayer").start();
    }

    /**
     * 用例消费者线程
     */
    private class ReplayerThread implements Runnable {

        private ReplayContext context;

        public ReplayerThread(ReplayContext context) {
            this.context = context;
        }

        @Override
        public void run() {
            try {
                replay();
            } catch (Exception e) {
                log.error("回放器线程异常", e);
                ReplayContextHolder.forceEnd();
            }
        }

        private void replay() throws InterruptedException {
            // (用例集未加载完成 || trace队列不是空的) 消费队列中的trace
            while (!context.isTracesLoadFinished() || !context.isTraceQueueEmpty()) {
                FlowReplayTrace trace = context.pollTrace();
                if (trace == null) {
                    continue;
                }
                try {
                    ReplayTraceResult traceResult = replayer.replay(trace);
                    context.offerResult(traceResult);
                } catch (Exception e) {
                    log.error("单个用例回放异常_traceId=" + trace.getTraceId(), e);
                }
            }
        }
    }
}
