package cn.com.duibaboot.ext.autoconfigure.flowreplay.replay.replayer;

import cn.com.duibaboot.ext.autoconfigure.flowreplay.FlowReplayException;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.FlowReplayTrace;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.SpringMvcFlowReplaySpan;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.replay.ReplayTraceContext;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.replay.ReplayTraceResult;
import lombok.extern.slf4j.Slf4j;
import org.apache.http.HttpEntity;
import org.apache.http.client.methods.*;
import org.apache.http.entity.ByteArrayEntity;
import org.apache.http.entity.ContentType;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.util.EntityUtils;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.http.HttpMethod;

import javax.annotation.Resource;
import java.io.IOException;
import java.util.Map;

/**
 * SpringMvc Controller的请求回放器
 * Created by guoyanfei .
 * 2019-02-26 .
 */
@Slf4j
public class SpringMvcReplayer implements Replayer {

    public static final String TRACE_ID = "Flow-Replay-Trace-Id";

    @Resource
    private CloseableHttpClient httpClient;

    @Value("${server.port}")
    private int httpServerPort;

    @Override
    public ReplayTraceResult replay(FlowReplayTrace trace) {
        SpringMvcFlowReplaySpan span = (SpringMvcFlowReplaySpan) trace.getMainSpan();

        String traceId = trace.getTraceId();

        // 在traceMap中加入当前trace，用于后续各种aop的读取
        FlowReplayTraceMapHolder.put(trace);
        ReplayTraceContextMapHolder.preset(traceId);

        HttpUriRequest request = buildHttpUriRequest(span);
        ReplayTraceContext context = null;
        byte[] replayResultValue = null;
        try (CloseableHttpResponse response = httpClient.execute(request)) {
            HttpEntity entity = response.getEntity();
            if (entity != null) {
                replayResultValue = EntityUtils.toByteArray(entity);
            }
        } catch (IOException e) {
            throw new FlowReplayException(e);
        } finally {
            context = ReplayTraceContextMapHolder.getAndRemove(traceId);
        }
        if (context == null) {
            throw new FlowReplayException("单个用例回放上下文丢失_traceId=" + traceId);
        }
        if (context.isErrorMarked()) {
            return ReplayTraceResult.failResult(trace, context.getErrorMsgType(), context.getErrorMsgDetail());
        }
        return ReplayTraceResult.successResult(trace, replayResultValue);
    }

    /**
     * 根据SpringMvcFlowReplaySpan构造回放的http请求
     * @param span
     * @return
     */
    private HttpUriRequest buildHttpUriRequest(SpringMvcFlowReplaySpan span) {
        RequestBuilder requestBuilder = RequestBuilder
                .create(this.getHttpMethod(span.getMethod()))
                .setUri("http://localhost:" + httpServerPort + span.getUri());

        // 设置httpHeaders
        Map<String, String> headers = span.getHeaders();
        if (headers != null) {
            for (Map.Entry<String, String> entry : headers.entrySet()) {
                requestBuilder.setHeader(entry.getKey(), entry.getValue());
            }
            // 回放请求需要加一个traceid的请求头，用于filter中获取
            requestBuilder.setHeader(TRACE_ID, span.getTraceId());
        }


        // 设置requestParameters
        Map<String, String[]> parameters = span.getParameters();
        if (parameters != null) {
            for (Map.Entry<String, String[]> entry : parameters.entrySet()) {
                for (String value : entry.getValue()) {
                    requestBuilder.addParameter(entry.getKey(), value);
                }
            }
        }

        requestBuilder.removeHeaders("Content-Length"); // 这个header不用设置，httpclient 自己会计算。设置之后会报错
        // 设置request body
        byte[] requestBody = span.getRequestBody();
        if (requestBody != null) {
            requestBuilder.setEntity(new ByteArrayEntity(requestBody, ContentType.APPLICATION_JSON));
        }

        return requestBuilder.build();
    }


    private String getHttpMethod(HttpMethod method) {
        switch (method) {
            case GET:
                return HttpGet.METHOD_NAME;
            case PUT:
                return HttpPut.METHOD_NAME;
            case POST:
                return HttpPost.METHOD_NAME;
            case DELETE:
                return HttpDelete.METHOD_NAME;
            case HEAD:
                return HttpHead.METHOD_NAME;
            case PATCH:
                return HttpPatch.METHOD_NAME;
            case OPTIONS:
                return HttpOptions.METHOD_NAME;
            case TRACE:
                return HttpTrace.METHOD_NAME;
        }
        throw new FlowReplayException("不存在的httpMethod");
    }
}
