package org.springframework.cloud.netflix.feign;

import cn.com.duiba.boot.cat.CatWithArgs;
import cn.com.duiba.boot.exception.BizException;
import cn.com.duiba.catmonitor.constants.CatConstants;
import cn.com.duibaboot.ext.autoconfigure.cat.context.CatContext;
import cn.com.duibaboot.ext.autoconfigure.core.rpc.RpcContext;
import cn.com.duibaboot.ext.autoconfigure.core.utils.CatUtils;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.*;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.replay.ReplayTraceContext;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.dianping.cat.Cat;
import com.dianping.cat.message.Event;
import com.dianping.cat.message.Message;
import com.dianping.cat.message.Transaction;
import com.dianping.cat.message.internal.AbstractMessage;
import com.esotericsoftware.kryo.KryoException;
import com.google.common.annotations.VisibleForTesting;
import com.netflix.hystrix.exception.HystrixBadRequestException;
import lombok.extern.slf4j.Slf4j;
import org.aopalliance.intercept.MethodInterceptor;

import java.lang.reflect.InvocationHandler;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.lang.reflect.Proxy;
import java.util.Iterator;
import java.util.Map;
import java.util.Objects;

/**
 * 拦截FeignClient的调用加入cat监控功能，以及拦截并注入当前调用的方法
 */
@Slf4j
public class CustomFeignClientFactoryBean extends FeignClientFactoryBean {

    //客户端应用名，CustomFeignClientsRegistrar.customFeignClientBeanDefinition会注入
//    @Value("${spring.application.name}")
    private String springApplicationName;

    @Override
    public Object getObject() throws Exception {
        //增加cat监控功能
        Class classType = super.getObjectType();
        Object feignClient = super.getObject();

        Object proxyInstance = proxyInstance(classType, feignClient);
        if (FlowReplayUtils.isReplayEnv()) {
            proxyInstance = Proxy.newProxyInstance(proxyInstance.getClass().getClassLoader(), new Class[] { classType }, new FlowReplayFeignClientInvocationHandler(proxyInstance));
        } else {
            proxyInstance = Proxy.newProxyInstance(proxyInstance.getClass().getClassLoader(), new Class[] { classType }, new FlowRecordFeignClientInvocationHandler(proxyInstance));
        }
        return proxyInstance;
    }

    private Object proxyInstance(Class classType, Object feignClient) {
        if (CatUtils.isCatClassExists()) {
            return Proxy.newProxyInstance(classType.getClassLoader(), new Class[] { classType }, new FeignClientInvocationHandlerWithCat(springApplicationName, feignClient, this.getName()));
        } else {
            return Proxy.newProxyInstance(classType.getClassLoader(), new Class[] { classType }, new FeignClientInvocationHandler(springApplicationName, feignClient, this.getName()));
        }
    }

    public static abstract class FlowRecordInvocationHandler implements InvocationHandler {

        protected final Object original;

        FlowRecordInvocationHandler(Object original) {
            this.original = original;
        }

        protected Object invokeMethodInner(Method method, Object target, Object[] args) throws Throwable {
            try {
                return method.invoke(target, args);
            } catch (Throwable e) {
                if (e instanceof InvocationTargetException) {
                    e = ((InvocationTargetException) e).getTargetException();
                }
                //通常这步拿出来的异常只会有以下2种：HystrixBadRequestException, FeignClientException
                if (e instanceof HystrixBadRequestException && e.getCause() != null && e.getCause() instanceof BizException) {
                    e = e.getCause();//如果cause是BizException，则应该把BizException提取出来抛出
                }
                throw e;
            }
        }
    }

    public static class FlowRecordFeignClientInvocationHandler extends FlowRecordInvocationHandler {

        FlowRecordFeignClientInvocationHandler(Object original) {
            super(original);
        }

        @Override
        public Object invoke(Object proxy, Method method, Object[] args) throws Throwable {
            Object ret = invokeMethodInner(method, original, args);
            try {
                if (FlowReplayTrace.isTraced()) {
                    FeignClientFlowReplaySpan span = FeignClientFlowReplaySpan.createSpan(method, args, ret);
                    span.setTraceId(FlowReplayTrace.getCurrentTraceId());
                    FlowReplayTrace.addSubSpan(span);
                }
            } catch (Throwable t) {
                log.error("feignClient录制异常", t);
            }
            return ret;
        }

    }

    public static class FlowReplayFeignClientInvocationHandler extends FlowRecordInvocationHandler {

        FlowReplayFeignClientInvocationHandler(Object original) {
            super(original);
        }

        @Override
        public Object invoke(Object proxy, Method method, Object[] args) throws Throwable {
            try {
                // 本机是回放环境 并且 当前请求是回放请求
                if (FlowReplayUtils.isReplayEnv() && ReplayTraceContext.isReplaying()) {
                    FlowReplaySpan span = ReplayTraceContext.pollSubSpan();

                    log.debug("FeignClient_poll_span_TraceId_{}_spanType_{}", ReplayTraceContext.getContextTraceId(), span != null ? span.getSpanType() : null);

                    // 增加了新的调用 || 调用内容有变动
                    if (span == null || SpanType.FEIGN_CLIENT != span.getSpanType()) {
                        String expert = SpanType.FEIGN_CLIENT.name();
                        String actual = span != null ? span.getSpanType().name() : null;
                        ReplayTraceContext.markError(FlowReplayErrorMsgTypeEnum.EM_001, expert, actual);
                        return method.invoke(original, args);
                    }

                    FeignClientFlowReplaySpan feignClientSpan = (FeignClientFlowReplaySpan) span;

                    // 判断录制的span的调用和本次调用是否一致，如果一致，那么直接mock录制的span的返回值
                    if (isRequestEqual(feignClientSpan, method, args)) {
                        try {
                            return feignClientSpan.getReturnValue();
                        } catch (KryoException e) {
                            ReplayTraceContext.markError(FlowReplayErrorMsgTypeEnum.EM_207, e.getMessage());
                        }
                    }
                }
            } catch (Throwable t) {
                log.error("feignClient回放异常", t);
            }
            return invokeMethodInner(method, original, args);
        }

        /**
         * 判断录制的span的调用和本次调用是否一致
         * 需要判断以下几点：
         * 1、类的全路径
         * 2、方法名称
         * 3、参数列表
         * 4、参数值列表
         * 5、返回值类型
         * @param feignClientSpan
         * @param currentMethod
         * @param currentArgs
         * @return
         */
        private boolean isRequestEqual(FeignClientFlowReplaySpan feignClientSpan, Method currentMethod, Object[] currentArgs) {
            if (!isTypeFullPathEqual(feignClientSpan, currentMethod)) {
                ReplayTraceContext.markError(FlowReplayErrorMsgTypeEnum.EM_205);
                return false;
            }
            if (!isApiNameEqual(feignClientSpan, currentMethod)) {
                String expert = feignClientSpan.getMethodName();
                String actual = currentMethod.getName();
                ReplayTraceContext.markError(FlowReplayErrorMsgTypeEnum.EM_201, expert, actual);
                return false;
            }
            if (!isParameterTypesEqual(feignClientSpan, currentMethod)) {
                String expert = FlowReplayUtils.stringArrayToString(feignClientSpan.getParameterTypes());
                String actual = FlowReplayUtils.classArrayToString(currentMethod.getParameterTypes());
                ReplayTraceContext.markError(FlowReplayErrorMsgTypeEnum.EM_202, expert, actual);
                return false;
            }
            try {
                if (!isParameterValuesEqual(feignClientSpan, currentArgs)) {
                    ReplayTraceContext.markError(FlowReplayErrorMsgTypeEnum.EM_203);
                    return false;
                }
            } catch (KryoException e) {
                ReplayTraceContext.markError(FlowReplayErrorMsgTypeEnum.EM_206, e.getMessage());
                return false;
            }

            if (!isReturnTypeEqual(feignClientSpan, currentMethod)) {
                ReplayTraceContext.markError(FlowReplayErrorMsgTypeEnum.EM_204);
                return false;
            }

            return true;
        }

        private boolean isTypeFullPathEqual(FeignClientFlowReplaySpan feignClientSpan, Method currentMethod) {
            String spanTypeFullPath = feignClientSpan.getTypeFullPath();
            String currentTypeFullPath = currentMethod.getDeclaringClass().getName();
            return Objects.equals(spanTypeFullPath, currentTypeFullPath);
        }

        private boolean isApiNameEqual(FeignClientFlowReplaySpan feignClientSpan, Method currentMethod) {
            String spanApiName = feignClientSpan.getApiName();
            String currentApiName = FlowReplayUtils.parseApiNameByMethod(currentMethod);
            return Objects.equals(spanApiName, currentApiName);
        }

        private boolean isParameterTypesEqual(FeignClientFlowReplaySpan feignClientSpan, Method currentMethod) {
            String[] spanParameterTypes = feignClientSpan.getParameterTypes();
            Class[] currentParameterTypes = currentMethod.getParameterTypes();

            int currentLength = currentParameterTypes.length;
            String[] currentParameterTypeNames = null;
            if (currentLength > 0) {
                currentParameterTypeNames = new String[currentLength];
                for (int i = 0; i < currentLength; i++) {
                    currentParameterTypeNames[i] = currentParameterTypes[i].getName();
                }
            }

            return Objects.deepEquals(spanParameterTypes, currentParameterTypeNames);
        }

        private boolean isParameterValuesEqual(FeignClientFlowReplaySpan feignClientSpan, Object[] currentArgs) {
            Object[] spanParameterValues = feignClientSpan.getParameterValues();
            if (Objects.deepEquals(spanParameterValues, currentArgs)) {
                return true;
            }
            for (int i = 0; i < spanParameterValues.length; i++) {
                Object spanObj = spanParameterValues[i];
                Object currentObj = currentArgs[i];
                if (!innerCompare(JSON.toJSON(spanObj), JSON.toJSON(currentObj))) {
                    return false;
                }
            }
            return true;
        }

        private boolean innerCompare(Object recordVal, Object replayVal) {
            if (recordVal == null && replayVal == null) {
                return true;
            }
            if (recordVal instanceof String && replayVal instanceof String) {
                return recordVal.equals(replayVal);
            }
            if (recordVal instanceof JSONArray && replayVal instanceof JSONArray) {
                return jsonArrayCompare((JSONArray) recordVal, (JSONArray) replayVal);
            } else if (recordVal instanceof JSONObject && replayVal instanceof JSONObject) {
                return jsonObjectCompare((JSONObject) recordVal, (JSONObject) replayVal);
            } else {
                return false;
            }
        }

        private boolean jsonArrayCompare(JSONArray recordArray, JSONArray replayArray) {
            if (recordArray.size() != replayArray.size()) {
                return false;
            }

            // 遍历录制返回值数组，每一项和回放返回值数组对应的项进行比对
            Iterator<Object> iterator = recordArray.iterator();
            for (int i = 0; iterator.hasNext(); i++) {
                Object recordValue = iterator.next();
                String recordValueStr = String.valueOf(recordValue);

                if (FlowReplayUtils.isJSONValid(recordValueStr)) { // 如果是jsonObj或者jsonArray，那么再判断内部的东西
                    return innerCompare(JSON.parse(recordValueStr), JSON.parse(String.valueOf(recordArray.get(i))));
                } else {    // 如果不是jsonObj或者jsonArray，那么直接进行值对比，如果值不一样，添加对比不一致的结果
                    if (!Objects.equals(recordValue, replayArray.get(i))) {
                        return false;
                    }
                }
            }
            return true;
        }

        private boolean jsonObjectCompare(JSONObject recordObj, JSONObject replayObj) {
            // 遍历录制返回值对象，每一个key的value和回放返回值对象对应的value进行比对
            for (Map.Entry<String, Object> entry : recordObj.entrySet()) {
                Object recordValue = entry.getValue();
                Object replayValue = replayObj.get(entry.getKey());
                String recordValueStr = String.valueOf(recordValue);
                String replayValueStr = String.valueOf(replayValue);

                if (FlowReplayUtils.isJSONValid(recordValueStr)) {  // 当前遍历到的value，是个jsonObj或者jsonArray，那么需要递归判断里面的内容
                    return innerCompare(JSON.parse(recordValueStr), JSON.parse(replayValueStr));
                } else {    // 当前遍历到的value，不是jsonObj或者jsonArray，那么直接进行值的比对，如果值不一样，添加对比不一致的结果
                    if (!Objects.equals(recordValue, replayValue)) {
                        return false;
                    }
                }
            }
            return true;
        }

        private boolean isReturnTypeEqual(FeignClientFlowReplaySpan feignClientSpan, Method currentMethod) {
            Object spanReturnType = feignClientSpan.getReturnType();
            return Objects.equals(spanReturnType, currentMethod.getReturnType().getName());
        }
    }

    public void setSpringApplicationName(String springApplicationName) {
        this.springApplicationName = springApplicationName;
    }

    /**
     * {@link MethodInterceptor}
     */
    @VisibleForTesting
    public static class FeignClientInvocationHandler implements InvocationHandler {

        protected final String springApplicationName;
        protected final String serverName;
        protected final Object original;

        FeignClientInvocationHandler(String springApplicationName, Object original, String serverName) {
            this.springApplicationName = springApplicationName;
            this.original = original;
            this.serverName = serverName;
        }

        @Override
        public Object invoke(Object proxy, Method method, Object[] args) throws Throwable {
            String methodName = method.getName();

            if ("equals".equals(methodName)
                    || "hashCode".equals(methodName)
                    || "toString".equals(methodName)
                    || method.isDefault()) {
                return invokeMethodInner(method, original, args);
            }

            RpcContext.getContext().setMethod(method);
            RpcContext.getContext().setInvokeArgs(args);
            RpcContext.getContext().setSourceServiceId(springApplicationName);
            RpcContext.getContext().setTargetServiceId(serverName);

            return invokeMethodInner(method, original, args);
        }

        protected Object invokeMethodInner(Method method, Object target, Object[] args) throws Throwable{
            try {
                return method.invoke(target, args);
            } catch (Throwable e) {
                if(e instanceof InvocationTargetException){
                    e = ((InvocationTargetException)e).getTargetException();
                }
                //通常这步拿出来的异常只会有以下2种：HystrixBadRequestException, FeignClientException
                if(e instanceof HystrixBadRequestException && e.getCause() != null && e.getCause() instanceof BizException){
                    e = e.getCause();//如果cause是BizException，则应该把BizException提取出来抛出
                }
                throw e;
            }
        }

    }

    /**
     * {@link MethodInterceptor}
     */
    @VisibleForTesting
    public static class FeignClientInvocationHandlerWithCat extends FeignClientInvocationHandler {

        FeignClientInvocationHandlerWithCat(String springApplicationName, Object original, String serverName) {
            super(springApplicationName, original, serverName);
        }

        @Override
        public Object invoke(Object proxy, Method method, Object[] args) throws Throwable {
            String methodName = method.getName();

            if ("equals".equals(methodName)
                    || "hashCode".equals(methodName)
                    || "toString".equals(methodName)
                    || method.isDefault()) {
                return invokeMethodInner(method, original, args);
            }

            RpcContext.getContext().setMethod(method);
            RpcContext.getContext().setInvokeArgs(args);
            RpcContext.getContext().setSourceServiceId(springApplicationName);
            RpcContext.getContext().setTargetServiceId(serverName);

            if(!CatUtils.isCatEnabled()){
                return invokeMethodInner(method, original, args);
            }

            return invokeWithCatTransaction(method, args);
        }

        private Object invokeWithCatTransaction(Method method, Object[] args) throws Throwable{
            String loggerName = resolveLoggerName(method, args);

            Transaction transaction = Cat.newTransaction(CatConstants.CROSS_CONSUMER, loggerName);

            try {
                CatContext context = new CatContext();

                Cat.logRemoteCallClient(context);
                setAttachment(context, method);

                Object result = invokeMethodInner(method, original, args);

                transaction.setStatus(Message.SUCCESS);
                return result;
            }catch(Throwable e){
                if(!(e instanceof BizException)) {
                    Cat.logError(e);//BizException是正常的业务异常，比如用户不存在，不应该记录到cat.
                    transaction.setStatus(e);
                }else{
                    //记录BizException
                    Event bizExceptionEvent = Cat.newEvent("BizException", String.format("BizException(message:%s,code:%s)(this exception will not trigger hystrix)",e.getMessage(),((BizException) e).getCode()));
//                    completeEvent(bizExceptionEvent);

                    AbstractMessage message = (AbstractMessage) bizExceptionEvent;
                    message.setCompleted(true);
                    message.setStatus(Event.SUCCESS);

                    transaction.addChild(bizExceptionEvent);

                    transaction.setStatus(Message.SUCCESS);
                }
                throw e;
            }finally {
                Event crossAppEvent =  Cat.newEvent(CatConstants.CONSUMER_CALL_APP, springApplicationName + "[" + RpcContext.getContext().getLocalAddr() + "]");
                Event crossServerEvent = Cat.newEvent(CatConstants.CONSUMER_CALL_SERVER, serverName + "[" + RpcContext.getContext().getRemoteAddr() + "]");
//            Event crossPortEvent = Cat.newEvent(CatConstants.CONSUMER_CALL_PORT,url.getPort()+"");
//            completeEvent(crossAppEvent);
//            completeEvent(crossPortEvent);
//            completeEvent(crossServerEvent);

                transaction.addChild(crossAppEvent);
//            transaction.addChild(crossPortEvent);
                transaction.addChild(crossServerEvent);

                CatUtils.completeTransaction(transaction);
            }
        }

        private String resolveLoggerName(Method method, Object[] args){//NOSONAR
            StringBuilder loggerName = new StringBuilder(method.getDeclaringClass().getSimpleName())
                    .append(".").append(method.getName());

            //注解了CatWithArgs表示cat中的监控名需要附加参数，详情见CatWithArgs类的注释
            if(method.isAnnotationPresent(CatWithArgs.class)){
                CatWithArgs catWithArgs = method.getAnnotation(CatWithArgs.class);
                int[] argIndexes = catWithArgs.argIndexes();
                if(argIndexes == null || argIndexes.length == 0
                        || args == null || args.length == 0){
                    return loggerName.toString();
                }

                StringBuilder sb = new StringBuilder();
                for(int argIdx : argIndexes){
                    if(argIdx >= 0 && args.length > argIdx) {
                        Object obj = args[argIdx];
                        sb.append(obj == null ? "" : obj.toString()).append(",");
                    }
                }

                if(sb.length() > 0){
                    sb.deleteCharAt(sb.length() - 1);
                    sb.insert(0, "(");
                    sb.append(")");
                }
                loggerName.append(sb);
            }
            return loggerName.toString();
        }

        private void setAttachment(CatContext context, Method method){
            RpcContext.getContext().setAttachment(Cat.Context.ROOT,context.getProperty(Cat.Context.ROOT));
            RpcContext.getContext().setAttachment(Cat.Context.CHILD,context.getProperty(Cat.Context.CHILD));
            RpcContext.getContext().setAttachment(Cat.Context.PARENT,context.getProperty(Cat.Context.PARENT));
            RpcContext.getContext().setAttachment(cn.com.duibaboot.ext.autoconfigure.core.Constants.X_RPC_CLIENT, springApplicationName);
        }

    }

}
