package cn.com.duibaboot.ext.autoconfigure.hazelcast;

import ch.qos.logback.classic.Level;
import cn.com.duiba.boot.utils.NetUtils;
import cn.com.duibaboot.ext.autoconfigure.cloud.netflix.eureka.DuibaEurekaAutoServiceRegistration;
import com.hazelcast.config.Config;
import com.hazelcast.config.DiscoveryStrategyConfig;
import com.hazelcast.core.HazelcastInstance;
import com.netflix.appinfo.ApplicationInfoManager;
import com.netflix.discovery.EurekaClient;
import lombok.extern.slf4j.Slf4j;
import org.springframework.boot.autoconfigure.AutoConfigureBefore;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.autoconfigure.hazelcast.HazelcastAutoConfiguration;
import org.springframework.boot.autoconfigure.hazelcast.HazelcastInstanceFactory;
import org.springframework.context.ApplicationContext;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;

import javax.annotation.Resource;
import java.util.HashMap;
import java.util.Map;

/**
 * 自动配置hazelcast的Config. 服务发现使用eureka.
 */
@Configuration
@Import(DuibaHazelcastPortConfiguration.class)
@ConditionalOnClass(EurekaClient.class)
@AutoConfigureBefore(HazelcastAutoConfiguration.class)
@Slf4j
public class DuibaHazelcastAutoConfiguration {

    static{
        ((ch.qos.logback.classic.Logger)log).setLevel(Level.INFO);
    }

    public static final String SPRING_APPLICATION_CONTEXT_KEY = "applicationContext";

    @Resource
    private ApplicationContext applicationContext;

    @Resource
    private EurekaClient eurekaClient;

    //强制注入DuibaEurekaAutoServiceRegistration，在构造HazelcastInstance之前确保已经注册到eurekaServer上。
    @Resource
    private DuibaEurekaAutoServiceRegistration duibaEurekaAutoServiceRegistration;

    @Resource
    private ApplicationInfoManager applicationInfoManager;

    @Bean
    public Config hazelcastConfig(DuibaHazelcastPortConfiguration duibaHazelcastPortConfiguration){
        Config config = new Config();
//        config.setInstanceName("fuck");
        config.getNetworkConfig().getJoin().getMulticastConfig().setEnabled(false);
        config.getNetworkConfig().getJoin().getTcpIpConfig().setEnabled(false);
        config.getNetworkConfig().getJoin().getAwsConfig().setEnabled(false);

        //获取当前节点的所有metadata信息
        Map<String, String> metadata = applicationInfoManager.getInfo().getMetadata();

        //设置当前node的属性,把当前节点的eureka metadata中所有信息都放进去
        config.getMemberAttributeConfig().getAttributes().putAll(metadata);

        //绑定spring的applicationContext，以供后续分布式线程池中使用
        config.getUserContext().put(SPRING_APPLICATION_CONTEXT_KEY, applicationContext);


        //强制hazelcast使用指定端口
        config.getNetworkConfig().setPort(duibaHazelcastPortConfiguration.getHazelcastPort());
        config.getNetworkConfig().setPortAutoIncrement(false);

        config.setProperty("hazelcast.discovery.enabled", "true");
        config.setProperty("hazelcast.logging.type", "slf4j");
        config.setProperty("hazelcast.local.localAddress", NetUtils.getLocalIp());//强制hazelcast使用这个ip来监听
        config.setProperty("hazelcast.socket.server.bind.any", "false");

        config.setProperty("hazelcast.heartbeat.failuredetector.type", "deadline");
        config.setProperty("hazelcast.heartbeat.interval.seconds", "5");
        config.setProperty("hazelcast.max.no.heartbeat.seconds", "20");

        Map<String, Comparable> eurekaStrategyProperties = new HashMap<>();

        EurekaOneDiscoveryStrategyFactory.setEurekaClient(eurekaClient);

        config.getNetworkConfig().getJoin().getDiscoveryConfig().addDiscoveryStrategyConfig(
                new DiscoveryStrategyConfig(EurekaOneDiscoveryStrategy.class.getName(), eurekaStrategyProperties)
        );

        return config;
    }

    @Configuration
    @ConditionalOnProperty(name={"org.springframework.boot.test.context.SpringBootTestContextBootstrapper", "duiba.test.mock.hazelcast"}, havingValue = "false", matchIfMissing = true)
    static class HazelcastInstanceConfiguration{
        //TODO hazelcast启动需耗时6-10秒，后续看看能否优化
        @Bean
        public HazelcastInstance hazelcastInstance(Config config) {
            checkVersion();
            long start = System.currentTimeMillis();
            HazelcastInstance instance =  new HazelcastInstanceFactory(config).getHazelcastInstance();
            long cost = System.currentTimeMillis() - start;
            log.info("hazelcast is initted in {}ms, clusterState:{}, version:{}, members:{}", cost, instance.getCluster().getClusterState(), instance.getCluster().getClusterVersion(), instance.getCluster().getMembers());

            return instance;
        }

        private void checkVersion(){
            Package pkg = HazelcastInstance.class.getPackage();
            String version = (pkg != null ? pkg.getImplementationVersion() : null);
            if(version.equals("3.11") || version.startsWith("3.11.")){
            }else{
                throw new IllegalStateException("请把hazelcast版本升级到3.11.*版本, 在你的gradle文件中加入这个配置即可升级：ext['hazelcast.version']='3.11' ");
            }
        }
    }

    /**
     * 一个mock实例，啥都不干，存在的意义是为了在执行单元测试时使用，加快单测速度
     */
    @Configuration
    @ConditionalOnProperty(name={"org.springframework.boot.test.context.SpringBootTestContextBootstrapper", "duiba.test.mock.hazelcast"}, havingValue = "true")
    static class MockHazelcastInstanceConfiguration{

        @Bean
        public HazelcastInstance hazelcastInstance(Config config) {
            log.info("检测到当前正在执行单元测试，mock 一个 HazelcastInstance ");

            return new MockHazelcastInstance();
        }
    }

    @Bean
    public HazelcastEndpoint hazelcastEndpoint(HazelcastInstance hazelcastInstance) {
        return new HazelcastEndpoint(hazelcastInstance);
    }

}
