package cn.com.duibaboot.ext.autoconfigure.httpclient.ssre;

import cn.com.duiba.boot.utils.AopTargetUtils;
import com.google.common.base.Throwables;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.config.BeanPostProcessor;

import java.lang.reflect.Field;
import java.util.*;

public class SsreBeanPostProcessor implements BeanPostProcessor {

    private Logger logger = LoggerFactory.getLogger(SsreBeanPostProcessor.class);

    private final Set<String> whitePackages = Sets.newLinkedHashSet();
    private final Map<Class, Class<? extends ClientWrapper>> acceptableClassesMap;
    {
        whitePackages.add("cn.com.duibaboot");
        whitePackages.add("org.springframework");
        whitePackages.add("com.netflix");
        whitePackages.add("org");
        whitePackages.add("io");

        Set<ClientWrapper> wrapperClassSet = Sets.newLinkedHashSet();
        wrapperClassSet.add(new SsreHttpClientWrapper());
        wrapperClassSet.add(new SsreHttpAsyncClientWrapper());
        wrapperClassSet.add(new SsreRestTemplateWrapper());
        wrapperClassSet.add(new SsreAsyncRestTemplateWrapper());

        Map<Class, Class<? extends ClientWrapper>> acceptableClassesMapTemp = new HashMap<>();
        for(ClientWrapper w : wrapperClassSet) {
            acceptableClassesMapTemp.put(w.acceptableClass(), w.getClass());
        }
        acceptableClassesMap = acceptableClassesMapTemp;
    }

    @Override
    public Object postProcessBeforeInitialization(Object bean, String beanName) throws BeansException {
        return bean;
    }

    @Override
    public Object postProcessAfterInitialization(Object bean, String beanName) throws BeansException {
        Object targetBean;//bean有可能被spring aop过，所以这里要先获取targetBean
        try {
            targetBean = AopTargetUtils.getTarget(bean);
        } catch (Exception e) {
            throw Throwables.propagate(e);
        }
        Class clazz = targetBean.getClass();
        if(!checkPackage(clazz.getPackage())){
            return bean;
        }
        List<Field> fields = getFieldList(clazz);
        for (Field field:fields){
            if (field.isAnnotationPresent(CanAccessInsideNetwork.class)){
                continue;
            }
            Class fieldType = field.getType();
            Class<? extends ClientWrapper> clientWrapperClass = getClientWrapperClass(fieldType);
            if(clientWrapperClass == null){
                continue;
            }

            try {
                field.setAccessible(true);
                Object obj = field.get(targetBean);
                if(obj == null || clientWrapperClass.isAssignableFrom(obj.getClass())){//防止重复代理而造成无限递归
                    continue;
                }
                ClientWrapper clientWrapper = clientWrapperClass.newInstance();
                if(clientWrapper.trySetClient(obj)){
                    field.set(targetBean, clientWrapper);
                }
            }catch (Exception e) {
                logger.warn("Client代理失败", e);
            }
        }
        return bean;
    }

    private Class<? extends ClientWrapper> getClientWrapperClass(Class clazz){
        if(clazz.isPrimitive()){
            return null;
        }
        for(Map.Entry<Class, Class<? extends ClientWrapper>> entry : acceptableClassesMap.entrySet()){
            Class c = entry.getKey();
            if(c.isAssignableFrom(clazz)){
                return entry.getValue();
            }
        }
        return null;
    }

    public void setWhitePackage(Set<String> whitePackage) {
        this.whitePackages.addAll(whitePackage);
    }

    private List<Field> getFieldList(Class clazz){
        if(null == clazz){
            return Collections.emptyList();
        }
        Field[] fields = clazz.getDeclaredFields();
        List<Field> fieldList = Lists.newArrayList(fields);
        Class superClass = clazz.getSuperclass();
        if(superClass.equals(Object.class)){
            return fieldList;
        }
        fieldList.addAll(getFieldList(superClass));
        return fieldList;
    }

    private boolean checkPackage(Package pakeage){
        if(pakeage==null){
            return false;
        }
        String packageName = pakeage.getName();
        for(String whitePackage:whitePackages){
            if(StringUtils.startsWith(packageName,whitePackage)){
                return false;
            }
        }
        return true;
    }

}
