package cn.com.duibaboot.ext.autoconfigure.threadpool.proxy;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.cloud.sleuth.instrument.async.SpanContinuingTraceCallable;

import java.util.concurrent.BlockingQueue;
import java.util.concurrent.Callable;

/**
 * Callable的包装类，可以监控Callable从提交到线程池到开始执行花了多久，如果过久则打印warn日志
 * Created by wenqi.huang on 2017/2/5.
 */
public class MonitorCallable implements Callable {
    private static final Logger logger = LoggerFactory.getLogger(MonitorCallable.class);

    private static final boolean IS_SLEUTH_CLASS_EXISTS;

    static{
        boolean isSleuthClassExists = false;
        try{
            Class.forName("org.springframework.cloud.sleuth.instrument.async.SpanContinuingTraceRunnable");
            isSleuthClassExists = true;
        }catch(Exception e){
        }
        IS_SLEUTH_CLASS_EXISTS = isSleuthClassExists;
    }

    private Callable callable;

    /**
     * 线程池中的queue，当超时的时候可以打印出大小
     */
    private BlockingQueue<Runnable> queue;

    //这个时间用于记录什么时候提交到线程池
    private final long submitTimeMillis;

    public MonitorCallable(Callable callable){
        this.callable = callable;
        submitTimeMillis = System.currentTimeMillis();
    }

    public MonitorCallable(Callable callable, BlockingQueue<Runnable> queue){
        this(callable);
        this.queue = queue;
    }

    public long getSubmitTimeMillis() {
        return submitTimeMillis;
    }

    public Callable getCallable() {
        return callable;
    }

    @Override
    public Object call() throws Exception {
        long waitTimeMillis = System.currentTimeMillis() - submitTimeMillis;
        if(waitTimeMillis >= MonitorRunnable.MAX_WAIT_TIME_MILLIS){
            if(queue == null) {
                logger.error("CallableClass:{},ThreadName:{} wait {} ms in queue, maybe corePoolSize is too small.", new Object[]{getRootCallable().getClass().getName(), Thread.currentThread().getName(), waitTimeMillis});
            }else{
                logger.error("CallableClass:{},ThreadName:{} wait {} ms in queue(queueSize:{}), maybe corePoolSize is too small.", new Object[]{getRootCallable().getClass().getName(), Thread.currentThread().getName(), waitTimeMillis, queue.size()});
            }
        }

        return callable.call();
    }

    /**
     * 获得最底层的callable对象
     * @return
     */
    protected Callable getRootCallable(){
        Callable rootCallable = getCallable();
        if(IS_SLEUTH_CLASS_EXISTS) {
            if (rootCallable instanceof SpanContinuingTraceCallable) {
                rootCallable = ((SpanContinuingTraceCallable) rootCallable).getDelegate();
            }
        }

        return rootCallable;
    }
}
