package cn.com.duibaboot.ext.autoconfigure.flowreplay.replay.aop;

import cn.com.duibaboot.ext.autoconfigure.flowreplay.*;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.replay.ReplayTraceContext;
import cn.com.duibaboot.ext.autoconfigure.javaagent.core.PluginException;
import cn.com.duibaboot.ext.autoconfigure.javaagent.core.interceptor.enhance.InstanceMethodsAroundInterceptor;
import cn.com.duibaboot.ext.autoconfigure.javaagent.core.interceptor.enhance.MethodInterceptResult;
import com.esotericsoftware.kryo.KryoException;
import lombok.extern.slf4j.Slf4j;

import java.lang.reflect.Method;

/**
 * 流量回放的redis方法拦截器
 * 把当前请求的方法名、参数列表，参数值列表和录制的trace中的进行对比，如果都相同，那么mock本次调用的返回值为trace中的返回值
 */
@Slf4j
public class ReplayJedisMethodInterceptor implements InstanceMethodsAroundInterceptor {

    /**
     * 可以回放
     * @param method
     * @param allArguments
     * @return
     */
    private boolean canReplay(Method method, Object[] allArguments) {
        // 当前不是回放的环境 或者 当前请求不是回放请求
        if (!FlowReplayUtils.isReplayEnv() || !ReplayTraceContext.isReplaying()) {
            return false;
        }

        if (FlowReplayConstants.REDIS_AOP_EXCLUDE_METHOD.contains(method.getName())) {
            return false;
        }

        if (FlowReplayUtils.isGlobalWhitelist(allArguments)) {
            return false;
        }
        return true;
    }

    @Override
    public void beforeMethod(Object obj, Method method, Object[] allArguments, Class<?>[] argumentsTypes, MethodInterceptResult result) {
        if (!canReplay(method, allArguments)) {
            return;
        }
        FlowReplaySpan span = ReplayTraceContext.pollSubSpan();

        log.debug("Redis回放_traceId={}_spanType={}_spanId={}", ReplayTraceContext.getContextTraceId(), span != null ? span.getSpanType() : null, span != null ? span.getSpanId() : null);

        // 增加了新的调用 || 调用内容有变动
        if (span == null || SpanType.REDIS != span.getSpanType()) {
            String expert = SpanType.REDIS.name();
            String actual = span != null ? span.getSpanType().name() : null;
            ReplayTraceContext.markError(FlowReplayErrorMsgTypeEnum.EM_001, expert, actual);
            throw new PluginException(ReplayTraceContext.getCompletedErrorMsg());
        }

        RedisFlowReplaySpan redisSpan = (RedisFlowReplaySpan) span;

        if (!FlowReplayUtils.isMethodEqual(redisSpan.getMethodName(), method)) {
            String expert = redisSpan.getMethodName();
            String actual = method.getName();
            ReplayTraceContext.markError(FlowReplayErrorMsgTypeEnum.EM_102, expert, actual);
            throw new PluginException(ReplayTraceContext.getCompletedErrorMsg());
        }

        if (!FlowReplayUtils.isArgumentsTypesEqual(redisSpan.getParameterTypes(), argumentsTypes)) {
            String expert = FlowReplayUtils.stringArrayToString(redisSpan.getParameterTypes());
            String actual = FlowReplayUtils.classArrayToString(argumentsTypes);
            ReplayTraceContext.markError(FlowReplayErrorMsgTypeEnum.EM_100, expert, actual);
            throw new PluginException(ReplayTraceContext.getCompletedErrorMsg());
        }

        try {
            if (!FlowReplayUtils.isArgumentsEqual(redisSpan.getParameterValues(), allArguments)) {
                ReplayTraceContext.markError(FlowReplayErrorMsgTypeEnum.EM_101);
                throw new PluginException(ReplayTraceContext.getCompletedErrorMsg());
            }
        } catch (KryoException e) {
            ReplayTraceContext.markError(FlowReplayErrorMsgTypeEnum.EM_103, e);
            throw new PluginException(ReplayTraceContext.getCompletedErrorMsg());
        }

        try {
            result.defineReturnValue(redisSpan.getReturnValue());
        } catch (KryoException e) {
            ReplayTraceContext.markError(FlowReplayErrorMsgTypeEnum.EM_104, e);
            throw new PluginException(ReplayTraceContext.getCompletedErrorMsg());
        }
    }

    @Override
    public Object afterMethod(Object zuperCall, Object obj, Method method, Object[] allArguments, Class<?>[] argumentsTypes, Object ret) {
        return ret;
    }

    @Override
    public void handleMethodException(Object objInst, Method method, Object[] allArguments, Class<?>[] argumentsTypes, Throwable t) {
        //do nothing
    }

}
