package cn.com.duibaboot.ext.autoconfigure.cat;

import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.reflect.MethodSignature;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cloud.sleuth.ErrorParser;
import org.springframework.cloud.sleuth.Span;
import org.springframework.cloud.sleuth.Tracer;

@Aspect
public class SleuthElasticSearchPlugin {

    @Autowired
    private Tracer tracer;
    @Autowired
    private ErrorParser errorParser;

    /**
     * 拦截ElasticsearchOperations类所有接口，记录执行耗时
     *
     * @param joinPoint
     * @return
     * @throws Throwable
     */
    @Around("execution(* org.springframework.data.elasticsearch.core.ElasticsearchOperations.*(..))")
    public Object springDataMongodbJoinPoint(ProceedingJoinPoint joinPoint) throws Throwable {
        MethodSignature signature = (MethodSignature) joinPoint.getSignature();
        String methodName = signature.getMethod().getName();

        if(!tracer.isTracing() || !tracer.getCurrentSpan().isExportable()){
            return joinPoint.proceed();
        }

        Span span = tracer.createSpan("elasticsearch:/"+methodName);
        StringBuilder stringBuilder = new StringBuilder();
        stringBuilder.append("ElasticsearchOperations").append(".").append(methodName);
        try {
            span.tag("elasticsearch.class_method", stringBuilder.toString());// get/...
            span.tag(Span.SPAN_LOCAL_COMPONENT_TAG_NAME, "ElasticsearchOperations");//本地组件名
            span.tag(Span.SPAN_PEER_SERVICE_TAG_NAME, "Elasticsearch");//远程服务名
            span.logEvent(Span.CLIENT_SEND);

            return joinPoint.proceed();
        } catch(Exception e){
            errorParser.parseErrorTags(span, e);
            throw e;
        } finally {
            span.logEvent(Span.CLIENT_RECV);
            this.tracer.close(span);
        }
    }


}
