package cn.com.duibaboot.ext.autoconfigure.perftest.guava;

import cn.com.duiba.boot.perftest.InternalPerfTestContext;
import cn.com.duibaboot.ext.autoconfigure.javaagent.core.interceptor.enhance.InstanceMethodsAroundInterceptor;
import cn.com.duibaboot.ext.autoconfigure.javaagent.core.interceptor.enhance.MethodInterceptResult;
import com.google.common.cache.Cache;
import com.google.common.cache.CacheStats;
import com.google.common.cache.LoadingCache;
import com.google.common.collect.ImmutableMap;
import org.springframework.util.Assert;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.Map;
import java.util.concurrent.Callable;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.ExecutionException;

import static com.google.common.base.Preconditions.checkNotNull;

/**
 * 拦截guava的CacheBuilder.build方法,进行两次build调用，从而生成两个Cache实例（一个给正常流量用，一个给压测流量用），并用一个新的Cache实例对它们进行包装。以防止压测流量影响正常流量
 */
public class GuavaCacheInterceptor implements InstanceMethodsAroundInterceptor {
    @Override
    public void beforeMethod(Object objInst, Method method, Object[] allArguments,
                             Class<?>[] argumentsTypes, MethodInterceptResult result) throws Throwable {
        //do nothing
    }

    @Override
    public Object afterMethod(Object zuperCall, Object objInst, Method method, Object[] allArguments,
                              Class<?>[] argumentsTypes, Object ret) throws Throwable {
        Assert.state("build".equals(method.getName()), "method name must be 'build', will not be here");

        if(method.getParameterCount() == 0){//CacheBuilder.build无参方法
            Cache cacheForNormal = (Cache)ret;
            Cache cacheForPerfTest = (Cache)((Callable<?>)zuperCall).call();
            return new AutoRoutingLocalCache(cacheForNormal, cacheForPerfTest);
        }else if(method.getParameterCount() == 1){//CacheBuilder.build带CacheLoader的方法
            LoadingCache cacheForNormal = (LoadingCache)ret;
            LoadingCache cacheForPerfTest = (LoadingCache)((Callable<?>)zuperCall).call();
            return new AutoRoutingLocalLoadingCache(cacheForNormal, cacheForPerfTest);
        }else {
            throw new IllegalStateException("will never be here");
        }
    }

    @Override
    public void handleMethodException(Object objInst, Method method, Object[] allArguments,
                                      Class<?>[] argumentsTypes, Throwable t) {
        //do nothing
    }

    /**
     * 内部封装了两个cache实例，分别给正常/压测流量使用，根据当前是否压测来决定路由给哪个cache
     * @param <K>
     * @param <V>
     */
    static class AutoRoutingLocalCache<K, V> implements Cache<K, V>{

        private Cache<K, V> cacheForNormal;//给正常请求使用的cache
        private Cache<K, V> cacheForPerfTest;//给压测请求实用的cache

        private AutoRoutingLocalCache(Cache<K, V> cacheForNormal, Cache<K, V> cacheForPerfTest){
            this.cacheForNormal = cacheForNormal;
            this.cacheForPerfTest = cacheForPerfTest;
        }

        protected Cache<K, V> determineCache(){
            return InternalPerfTestContext.isCurrentInPerfTestMode() ? cacheForPerfTest : cacheForNormal;
        }

        // Cache methods

        @Override
//        @Nullable
        public V getIfPresent(Object key) {
            return determineCache().getIfPresent(key);
        }

        @Override
        public V get(K key, final Callable<? extends V> valueLoader) throws ExecutionException {
            checkNotNull(valueLoader);
            return determineCache().get(key, valueLoader);
        }

        @Override
        public ImmutableMap<K, V> getAllPresent(Iterable<?> keys) {
            return determineCache().getAllPresent(keys);
        }

        @Override
        public void put(K key, V value) {
            determineCache().put(key, value);
        }

        @Override
        public void putAll(Map<? extends K, ? extends V> m) {
            determineCache().putAll(m);
        }

        @Override
        public void invalidate(Object key) {
            checkNotNull(key);
            determineCache().invalidate(key);
        }

        @Override
        public void invalidateAll(Iterable<?> keys) {
            determineCache().invalidateAll(keys);
        }

        @Override
        public void invalidateAll() {
            determineCache().invalidateAll();
        }

        @Override
        public long size() {
            return determineCache().size();
        }

        @Override
        public ConcurrentMap<K, V> asMap() {
            return determineCache().asMap();
        }

        @Override
        public CacheStats stats() {
            return determineCache().stats();
        }

        @Override
        public void cleanUp() {
            determineCache().cleanUp();
        }

        Object writeReplace() {
            Cache cache = determineCache();
            try {
                Method method = cache.getClass().getDeclaredMethod("writeReplace");
                method.setAccessible(true);
                return method.invoke(cache);
            } catch (NoSuchMethodException|InvocationTargetException|IllegalAccessException e) {
                throw new RuntimeException(e);
            }
        }
    }

    /**
     * 内部封装了两个cache实例，分别给正常/压测流量使用，根据当前是否压测来决定路由给哪个cache
     * @param <K>
     * @param <V>
     */
    static class AutoRoutingLocalLoadingCache<K, V> extends AutoRoutingLocalCache<K, V> implements LoadingCache<K, V>{

        private AutoRoutingLocalLoadingCache(LoadingCache<K, V> cacheForNormal, LoadingCache<K, V> cacheForPerfTest){
            super(cacheForNormal, cacheForPerfTest);
        }

        protected LoadingCache<K, V> determineCache(){
            return (LoadingCache)super.determineCache();
        }

        // LoadingCache methods

        @Override
        public V get(K key) throws ExecutionException {
            return determineCache().get(key);
        }

        @Override
        public V getUnchecked(K key) {
            return determineCache().getUnchecked(key);
        }

        @Override
        public ImmutableMap<K, V> getAll(Iterable<? extends K> keys) throws ExecutionException {
            return determineCache().getAll(keys);
        }

        @Override
        public void refresh(K key) {
            determineCache().refresh(key);
        }

        @Override
        public final V apply(K key) {
            return determineCache().apply(key);
        }

        Object writeReplace() {
            LoadingCache cache = determineCache();
            try {
                Method method = cache.getClass().getDeclaredMethod("writeReplace");
                method.setAccessible(true);
                return method.invoke(cache);
            } catch (NoSuchMethodException|InvocationTargetException|IllegalAccessException e) {
                throw new RuntimeException(e);
            }
        }
    }
}
