package cn.com.duibaboot.ext.autoconfigure.flowreplay.replay.aop;

import cn.com.duibaboot.ext.autoconfigure.flowreplay.*;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.replay.ReplayCloseableHttpResponse;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.replay.ReplayTraceContext;
import cn.com.duibaboot.ext.autoconfigure.javaagent.core.PluginException;
import cn.com.duibaboot.ext.autoconfigure.javaagent.core.interceptor.enhance.InstanceMethodsAroundInterceptor;
import cn.com.duibaboot.ext.autoconfigure.javaagent.core.interceptor.enhance.MethodInterceptResult;
import com.google.common.base.Preconditions;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.http.ProtocolVersion;
import org.apache.http.StatusLine;
import org.apache.http.client.methods.HttpRequestWrapper;
import org.apache.http.message.BasicStatusLine;

import java.lang.reflect.Method;
import java.net.URI;
import java.util.Arrays;
import java.util.List;
import java.util.Map;

/**
 * Created by guoyanfei .
 * 2019-07-18 .
 */
@Slf4j
public class ReplayHttpClientMethodInterceptor implements InstanceMethodsAroundInterceptor {

    /**
     * 可以回放
     * @param method
     * @param allArguments
     * @return
     */
    private boolean canReplay(Method method, Object[] allArguments) {
        if (!"execute".equals(method.getName())) {
            return false;
        }
        // 当前不是回放的环境 或者 当前请求不是回放请求
        if (!FlowReplayUtils.isReplayEnv() || !ReplayTraceContext.isReplaying()) {
            return false;
        }
        return true;
    }

    @Override
    public void beforeMethod(Object obj, Method method, Object[] allArguments, Class<?>[] argumentsTypes, MethodInterceptResult result) throws Throwable {
        if (!canReplay(method, allArguments)) {
            return;
        }

        FlowReplaySpan span = ReplayTraceContext.pollSubSpan();

        log.debug("httpClient回放_traceId={}_spanType={}_spanId={}", ReplayTraceContext.getContextTraceId(), span != null ? span.getSpanType() : null, span != null ? span.getSpanId() : null);

        // 增加了新的调用 || 调用内容有变动
        if (span == null || SpanType.HTTP_CLIENT != span.getSpanType()) {
            String expert = SpanType.HTTP_CLIENT.name();
            String actual = span != null ? span.getSpanType().name() : null;
            ReplayTraceContext.markError(FlowReplayErrorMsgTypeEnum.EM_001, expert, actual);
            throw new PluginException(ReplayTraceContext.getCompletedErrorMsg());
        }

        HttpClientFlowReplaySpan httpClientSpan = (HttpClientFlowReplaySpan) span;

        HttpRequestWrapper httpRequestWrapper = (HttpRequestWrapper) allArguments[1];
        URI uri = httpRequestWrapper.getURI();
        String currentHttpMethod = httpRequestWrapper.getMethod();
        String currentUrl = HttpClientFlowReplaySpan.parseUrl(uri);
        Map<String, List<String>> currentParameters = HttpClientFlowReplaySpan.parseRequestParameters(uri);
        byte[] currentRequestBody = HttpClientFlowReplaySpan.parseRequestBody(httpRequestWrapper);

        if (!httpClientSpan.getMethod().equals(currentHttpMethod)) {
            String expert = httpClientSpan.getMethod();
            String actual = currentHttpMethod;
            ReplayTraceContext.markError(FlowReplayErrorMsgTypeEnum.EM_1011, expert, actual);
            throw new PluginException(ReplayTraceContext.getCompletedErrorMsg());
        }

        if (!httpClientSpan.getUrl().equals(currentUrl)) {
            String expert = httpClientSpan.getUrl();
            String actual = currentUrl;
            ReplayTraceContext.markError(FlowReplayErrorMsgTypeEnum.EM_1012, expert, actual);
            throw new PluginException(ReplayTraceContext.getCompletedErrorMsg());
        }

        if (!isParametersEqual(httpClientSpan.getRequestParameters(), currentParameters)) {
            String expert = httpClientSpan.getRequestParameters() != null ? httpClientSpan.getRequestParameters().toString() : null;
            String actual = currentParameters != null ? currentParameters.toString() : null;
            ReplayTraceContext.markError(FlowReplayErrorMsgTypeEnum.EM_1013, expert, actual);
            throw new PluginException(ReplayTraceContext.getCompletedErrorMsg());
        }

        if (!Arrays.equals(httpClientSpan.getRequestBody(), currentRequestBody)) {
            ReplayTraceContext.markError(FlowReplayErrorMsgTypeEnum.EM_1014);
            throw new PluginException(ReplayTraceContext.getCompletedErrorMsg());
        }

        ProtocolVersion version = new ProtocolVersion(httpClientSpan.getProtocol(), httpClientSpan.getMajor(), httpClientSpan.getMinor());
        StatusLine statusLine = new BasicStatusLine(version, httpClientSpan.getStatusCode(), httpClientSpan.getReasonPhrase());
        result.defineReturnValue(new ReplayCloseableHttpResponse(httpClientSpan.getResponseBody(), httpClientSpan.getResponseHeaders(), statusLine));
    }


    @Override
    public Object afterMethod(Object zuperCall, Object obj, Method method, Object[] allArguments, Class<?>[] argumentsTypes, Object ret) throws Throwable {
        return ret;
    }

    @Override
    public void handleMethodException(Object obj, Method method, Object[] allArguments, Class<?>[] argumentsTypes, Throwable t) {
        // do nothing
    }

    private boolean isParametersEqual(Map<String, List<String>> recordParameters, Map<String, List<String>> currentParameters) {
        Preconditions.checkNotNull(recordParameters, "recordParameters can't be null");
        Preconditions.checkNotNull(currentParameters, "currentParameters can't be null");
        if (recordParameters.size() != currentParameters.size()) {
            return false;
        }
        for (Map.Entry<String, List<String>> entry : recordParameters.entrySet()) {
            String k = entry.getKey();
            List<String> v = entry.getValue();
            List<String> cv = currentParameters.get(k);
            if (v == null && cv == null) {
                continue;
            }
            if (v == null || cv == null) {
                return false;
            }
            if (!CollectionUtils.isEqualCollection(v, cv)) {
                return false;
            }
        }
        return true;
    }
}
