package org.springframework.cloud.netflix.feign;

import cn.com.duiba.boot.netflix.feign.AdvancedFeignClient;
import com.fasterxml.jackson.databind.JavaType;
import com.fasterxml.jackson.databind.ObjectMapper;
import feign.Util;
import feign.hystrix.FallbackFactory;
import org.apache.commons.collections.set.SynchronizedSet;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.slf4j.bridge.SLF4JBridgeHandler;
import org.springframework.beans.factory.BeanClassLoaderAware;
import org.springframework.beans.factory.config.BeanDefinitionHolder;
import org.springframework.beans.factory.support.AbstractBeanDefinition;
import org.springframework.beans.factory.support.BeanDefinitionBuilder;
import org.springframework.beans.factory.support.BeanDefinitionReaderUtils;
import org.springframework.beans.factory.support.BeanDefinitionRegistry;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ResourceLoaderAware;
import org.springframework.context.annotation.ClassPathScanningCandidateComponentProvider;
import org.springframework.context.annotation.ImportBeanDefinitionRegistrar;
import org.springframework.core.io.ResourceLoader;
import org.springframework.core.type.AnnotationMetadata;
import org.springframework.core.type.classreading.MetadataReader;
import org.springframework.core.type.classreading.MetadataReaderFactory;
import org.springframework.core.type.filter.TypeFilter;
import org.springframework.util.StringUtils;

import java.io.IOException;
import java.lang.reflect.*;
import java.util.*;
import java.util.concurrent.atomic.AtomicReference;
import java.util.logging.Level;

/**
 *
 * 最后一个方法的作用：排除当前工程中的FeignClient（排除规则：判断FeignClient中的name和spring.application.name属性如果一样则排除，spring.application.name属性必须声明在bootstrap.properties文件中）
 *
 * Created by wenqi.huang on 2017/7/18.
 */
public class CustomFeignClientsRegistrar extends BaseFeignClientsRegistrar implements ImportBeanDefinitionRegistrar,
        ResourceLoaderAware, BeanClassLoaderAware {

    protected final Logger logger = LoggerFactory.getLogger(getClass());

    public static final int ADVANCED_FEIGN_CLIENT_ANNO_MIN_PACKAGE_LEVEL = 3;

    /**
     * 所有启用的FeignClient的服务器名
     */
    private static final Set<String> enabledFeignClientNames = SynchronizedSet.decorate(new HashSet<>());

    private ResourceLoader resourceLoader;
    //仅用于校验FeignClient/AdvancedFeignClient的参数和返回类型是否支持json反序列化
    private ObjectMapper objectMapperForValidate;
    private Boolean isLoadThisProjectFeignClients;
    private String applicationName;

    public static Set<String> getEnabledFeignClientNames(){
        return Collections.unmodifiableSet(enabledFeignClientNames);
    }

    @Override
    public void setResourceLoader(ResourceLoader resourceLoader) {
        super.setResourceLoader(resourceLoader);
        this.resourceLoader = resourceLoader;
        this.objectMapperForValidate = new ObjectMapper();

        if(!(resourceLoader instanceof ApplicationContext)){
            logger.warn("resourceLoader is not instance of ConfigurableWebApplicationContext, will load all FeignClient including the FeignClient in current project");
        }else {
            ApplicationContext context = ((ApplicationContext) resourceLoader);

            applicationName = context.getEnvironment().getProperty("spring.application.name");
            if (applicationName == null || applicationName.isEmpty()) {
                throw new IllegalStateException("property[spring.application.name] is not exist");
            }
        }
    }

    /**
     * 在FeignClient扫描器中增加处理逻辑：排除当前工程中的FeignClient（排除规则：判断FeignClient中的name和spring.application.name属性如果一样则排除，spring.application.name属性必须声明在bootstrap.properties文件中）
     * @return
     */
    protected ClassPathScanningCandidateComponentProvider getScanner() {
        ClassPathScanningCandidateComponentProvider scanner = super.getScanner();

        postHandle(scanner);

        return scanner;
    }

    private boolean isLoadThisProjectFeignClients(){
        if(this.isLoadThisProjectFeignClients == null){
            ApplicationContext context = ((ApplicationContext) resourceLoader);

            //当测试类上注解了@SpringBootTest时，spring会自动加上属性:org.springframework.boot.test.context.SpringBootTestContextBootstrapper=true
            String isInUnitTest = context.getEnvironment().getProperty("org.springframework.boot.test.context.SpringBootTestContextBootstrapper");
            //duiba.feign.load-this-project-feign-clients 这个属性用于指定是否加载当前项目中的FeignClients(其他项目中的仍然会加载),默认为false，只有单测时指定该参数才会生效
            String isLoadThisProjectFeignClients = context.getEnvironment().getProperty("duiba.feign.load-this-project-feign-clients");
            //判断当前是否在执行单元测试,执行单测时也加载当前项目中声明的FeignClient
            if("true".equals(isInUnitTest) && "true".equals(isLoadThisProjectFeignClients)){
                this.isLoadThisProjectFeignClients = true;
            }else{
                this.isLoadThisProjectFeignClients = false;
            }
        }
        return this.isLoadThisProjectFeignClients;
    }

    /**
     * 排除当前工程中的FeignClient（排除规则：判断FeignClient中的name和spring.application.name属性如果一样则排除，spring.application.name属性必须声明在bootstrap.properties文件中）
     */
    protected void postHandle(ClassPathScanningCandidateComponentProvider scanner) {
        if(!(resourceLoader instanceof ApplicationContext)){
            logger.warn("resourceLoader is not instance of ConfigurableWebApplicationContext, will load all FeignClient");
            return;
        }
        ApplicationContext context = ((ApplicationContext) resourceLoader);

        boolean isLoadThisProjectFeignClients = isLoadThisProjectFeignClients();
        if(isLoadThisProjectFeignClients){
            return;
        }

        scanner.addExcludeFilter(new TypeFilter() {
            @Override
            public boolean match(MetadataReader metadataReader, MetadataReaderFactory metadataReaderFactory) throws IOException {
                Map<String, Object> map = metadataReader.getAnnotationMetadata().getAnnotationAttributes(FeignClient.class.getName());
                if(metadataReader.getClassMetadata().isAnnotation() //比如AdvancedFeignClient
                        || metadataReader.getClassMetadata().getClassName().endsWith(".package-info")){//package-info.java
                    return true;
                }

                if(map == null || map.isEmpty()){
                    return false;//不过滤掉
                }

                //判断如果FeignClient的name没有配置，到当前package-info中找，如果还没有，则到上级包找，直到找到为止。
                customClientName(map, metadataReader.getAnnotationMetadata().getClassName());

                String name = (String)map.get("name");

                return name.equals(applicationName);
            }
        });
    }

    protected void registerFeignClient(BeanDefinitionRegistry registry,
                                       AnnotationMetadata annotationMetadata, Map<String, Object> attributes) {
        validateMethodArgsAndReturnType(annotationMetadata.getClassName());
        super.registerFeignClient(registry, annotationMetadata,  attributes);

        registerFeignClientFallBack(registry, annotationMetadata,  attributes);
    }

    /**
     * 注册FeignClient/AdvancedFeignClient的降级bean，如果FeignClient的fallback或者fallbackFactory非空，则注册对应的class到spring上下文中。
     *
     * @param registry
     * @param annotationMetadata
     * @param attributes
     */
    private void registerFeignClientFallBack(BeanDefinitionRegistry registry,
                                             AnnotationMetadata annotationMetadata, Map<String, Object> attributes){
        //Feign会优先使用fallback，所有如果fallback和fallbackFactory都提供了，只声明fallback即可
        Class fallbackClazz = (Class)attributes.get("fallback");
        Class fallbackFactoryClazz = (Class)attributes.get("fallbackFactory");
        Class clazzToRegister = null;
        Class feignClientClass;
        try {
            feignClientClass = Class.forName(annotationMetadata.getClassName());
        }catch(ClassNotFoundException e){
            throw new RuntimeException(e);
        }
        if(fallbackClazz != null && !fallbackClazz.equals(void.class)){
            clazzToRegister = fallbackClazz;
            if(!feignClientClass.isAssignableFrom(clazzToRegister)){
                throw new IllegalStateException(
                        String.format(
                                "Incompatible fallback instance for feign client %s. instances of '%s', but should instances of '%s'",
                                feignClientClass.getName(), clazzToRegister.getName(), feignClientClass.getName()));
            }
        }
        else if(fallbackFactoryClazz != null && !fallbackFactoryClazz.equals(void.class)){
            clazzToRegister = fallbackFactoryClazz;
            if(!FallbackFactory.class.isAssignableFrom(fallbackFactoryClazz)){
                throw new IllegalStateException(String.format("%s 's fallbackFactory is invalid,must implements feign.hystrix.FallbackFactory", annotationMetadata.getClassName()));
            }else{//判断FallbackFactory生产的fallback是feignClientClass的子类
                Type[] intls = fallbackFactoryClazz.getGenericInterfaces();
                for(Type type: intls){
                    if(type instanceof ParameterizedType){
                        ParameterizedType ptype = (ParameterizedType)type;
                        if(!(ptype.getRawType() instanceof Class) && !FallbackFactory.class.isAssignableFrom((Class)ptype.getRawType())){
                            continue;
                        }
                        Class fallbackClazzProduces = (Class)ptype.getActualTypeArguments()[0];
                        if(!feignClientClass.isAssignableFrom(fallbackClazzProduces)){
                            throw new IllegalStateException(
                                    String.format(
                                            "Incompatible fallbackFactory instance for feign client %s. Factory produces instances of '%s', but should produce instances of '%s'",
                                            feignClientClass.getName(), fallbackClazzProduces.getName(), feignClientClass.getName()));
                        }
                    }
                }
            }
        }

        if(clazzToRegister != null) {
            BeanDefinitionBuilder definition = BeanDefinitionBuilder
                    .genericBeanDefinition(clazzToRegister);
            validate(attributes);

            definition.setAutowireMode(AbstractBeanDefinition.AUTOWIRE_NO);
//            definition.setDependencyCheck(AbstractBeanDefinition.DEPENDENCY_CHECK_NONE);

            AbstractBeanDefinition beanDefinition = definition.getBeanDefinition();
            beanDefinition.setPrimary(false);
            //指定如下这行属性后，如果存在多个同类bean，此bean的优先级最低，使用@Autowired、@Resource注解的时候不会被注入到其他bean, 除非明确指定beanName
            beanDefinition.setAutowireCandidate(false);

            String classSimpleName = clazzToRegister.getSimpleName();
            String beanName = classSimpleName.substring(0,1).toLowerCase()
                    + (classSimpleName.length() > 1 ? classSimpleName.substring(1) : "");
            BeanDefinitionHolder holder = new BeanDefinitionHolder(beanDefinition, beanName);
            BeanDefinitionReaderUtils.registerBeanDefinition(holder, registry);
        }
    }

    /**
     * 校验FeignClient或AdvancedFeignClient的参数和返回类型是否能被JSON序列化支持,如果不支持，则报错并禁止启动
     * （不是很有用，对于List&lt;T&gt;/List&lt;?&gt;/List&lt;? extends UserDto&gt;/List&lt;?&gt;/List等类型也会认为允许反序列化)
     * @param feignClientClassName
     */
    private void validateMethodArgsAndReturnType(String feignClientClassName){
        Class feignClientClass;
        try {
            feignClientClass = Class.forName(feignClientClassName);
        }catch(ClassNotFoundException e){
            throw new RuntimeException(e);
        }

        for(Method method : feignClientClass.getMethods()){
            if (method.getDeclaringClass() == Object.class ||
                    (method.getModifiers() & Modifier.STATIC) != 0 ||
                    Util.isDefault(method)) {
                continue;
            }

            //判断参数能否被json反序列化
            int paramIndex = 0;
            for(Type parameterType : method.getGenericParameterTypes()){
                JavaType javaType = objectMapperForValidate.getTypeFactory().constructType(parameterType);

                AtomicReference<Throwable> ref = new AtomicReference<>();
                boolean support = objectMapperForValidate.canDeserialize(javaType, ref);
                if(!support){
                    throw new IllegalStateException(String.format("parameter [%s](paramterIndex:%d) of method:[%s] can not be deserialized by json", parameterType.getTypeName(), paramIndex, method.toString()), ref.get());
                }
                paramIndex++;
            }

            //判断返回类型是否能被json反序列化
            Type returnType = method.getGenericReturnType();
            if(returnType == void.class){
                continue;
            }
            JavaType javaType = objectMapperForValidate.getTypeFactory().constructType(returnType);

            AtomicReference<Throwable> ref = new AtomicReference<>();
            boolean support = objectMapperForValidate.canDeserialize(javaType, ref);
            if(!support){
                throw new IllegalStateException(String.format("returnType [%s] of method:[%s] can not be deserialized by json", returnType.getTypeName(), method.toString()), ref.get());
            }
        }
    }

    /**
     * 判断如果FeignClient的name没有配置，到当前package-info中找，如果还没有，则到上级包找，直到找到为止, 另外此注解必须注解在至少包含了三个层级的包上
     *
     * @param attributes
     * @param feignClientClassName
     */
    protected void customClientName(Map<String, Object> attributes, String feignClientClassName){
        if (attributes == null) {
            return;
        }
        String value = (String) attributes.get("value");
        if (!StringUtils.hasText(value)) {
            value = (String) attributes.get("name");
        }
        if (!StringUtils.hasText(value)) {
            value = (String) attributes.get("serviceId");
        }

        if (!StringUtils.hasText(value)) {
            int idx = feignClientClassName.lastIndexOf(".");
            if(idx != -1) {
                String currentPackage = feignClientClassName.substring(0, idx);
                value = searchAndSetClientNameFromPackageInfo(currentPackage);
                if(value != null){
                    attributes.put("name" ,value);
                    attributes.put("value" ,value);
                }
            }
        }

        if(value != null && !value.isEmpty()) {
            //当指定了不加载当前工程的FeignClient时，不把对应名字加到【启用的FeignClient服务器列表】中
            if(isLoadThisProjectFeignClients() || !org.apache.commons.lang.StringUtils.equals(value, applicationName)) {
                enabledFeignClientNames.add(value);
            }
        }
    }

    private String searchAndSetClientNameFromPackageInfo(String packageName){
        if(packageName == null){
            return null;
        }
        try {
            //如果还没有加载过某个包下的类，则Package肯定找不到，所以这里先初始化一下
            Class.forName(packageName + ".package-info");
        } catch (ClassNotFoundException e) {
            //Ignore
        }
        Package pkg = Package.getPackage(packageName);
        if(pkg == null){
            //递归地到上级package寻找
            return searchAndSetClientNameFromPackageInfo(parentPackage(packageName));
        }
        AdvancedFeignClient feignClientAnno = pkg.getAnnotation(AdvancedFeignClient.class);
        if(feignClientAnno == null ||
                (!StringUtils.hasText(feignClientAnno.name()) && !StringUtils.hasText(feignClientAnno.value()))){
            //递归地到上级package寻找
            return searchAndSetClientNameFromPackageInfo(parentPackage(packageName));
        }else{
            int packageLevel = org.apache.commons.lang3.StringUtils.split(packageName, ".").length;
            if(packageLevel < ADVANCED_FEIGN_CLIENT_ANNO_MIN_PACKAGE_LEVEL){
                //AdvancedFeignClient如果标注在package上，则这个package必须至少有三个层级,比如com.tuia.activitycenter,就是说只要要在项目级别，每个项目的package都不一样，以避免冲突.
                throw new IllegalStateException(String.format("@AdvancedFeignClient must be annotated on a package have at least %d levels, current package is : %s, level is: %d", ADVANCED_FEIGN_CLIENT_ANNO_MIN_PACKAGE_LEVEL, packageName, packageLevel));
            }
            return StringUtils.hasText(feignClientAnno.name()) ? feignClientAnno.name() : feignClientAnno.value();
        }
    }

    private String parentPackage(String packageName){
        int idx = packageName.lastIndexOf(".");
        if(idx == -1){
            return null;
        }
        return packageName.substring(0, idx);
    }

    private static final java.util.logging.Logger logger1 = java.util.logging.Logger.getLogger("cn.com.duiba.A");

    protected static final Logger logger2 = LoggerFactory.getLogger("cn.com.duiba.A");
    public static void main(String[] args) {
        logger1.log(Level.SEVERE, "hello1");
        SLF4JBridgeHandler.removeHandlersForRootLogger();
        SLF4JBridgeHandler.install();
        logger1.log(Level.SEVERE, "hello2");
        logger2.error("fsdfsd");
    }

}
