package cn.com.duibaboot.ext.autoconfigure.graceclose;

import ch.qos.logback.classic.Level;
import ch.qos.logback.classic.LoggerContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;
import org.springframework.context.SmartLifecycle;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;

import java.util.concurrent.TimeUnit;

@Configuration
public class GracefulCloseAutoConfiguration {
    private static final Logger logger = LoggerFactory.getLogger(GracefulCloseAutoConfiguration.class);

    @Autowired
    private ApplicationContext applicationContext;

    @Bean
    public GracefulCloseLifeCycle gracefulCloseLifeCycle(){
//        String[] profiles = applicationContext.getEnvironment().getActiveProfiles();
//        boolean waitNginxToCheck = true;
//        for(String p : profiles){
//            if("dev".equals(p) || "test".equals(p) || "daily".equals(p)){
//                waitNginxToCheck = false;
//                break;
//            }
//        }
//
//        int sleepSeconds = 0;
//        if(waitNginxToCheck) {
//            //等待时间设置为6秒，让SLB或者nginx健康检查有足够的时间来调用/monitor/check接口发现服务不可用，从而关闭对本机的流量.(SLB/nginx一般配置每隔2秒检查一次/monitor/check)
//            //需要配合运维的脚本生效，运维kill pid后需要等待16秒后再强杀应用。
//            sleepSeconds = 6;
//        }

        //等待时间设置为6秒，让SLB或者nginx健康检查有足够的时间来调用/monitor/check接口发现服务不可用，从而关闭对本机的流量.(SLB/nginx一般配置每隔2秒检查一次/monitor/check)
        //需要配合运维的脚本生效，运维kill pid后需要等待16秒后再强杀应用。
        int sleepSeconds = 6;

        //当测试类上注解了@SpringBootTest时，spring会自动加上属性:org.springframework.boot.test.context.SpringBootTestContextBootstrapper=true
        String isInUnitTest = applicationContext.getEnvironment().getProperty("org.springframework.boot.test.context.SpringBootTestContextBootstrapper");
        //判断当前是否在执行单元测试,执行单测为了加快速度，不sleep
        if("true".equals(isInUnitTest)){
            sleepSeconds = 0;
        }

        return new GracefulCloseLifeCycle(sleepSeconds);
    }

    /**
     * 这个类的stop方法会被spring自动调用，spring会在取消注册eureka/consul后进入本类的stop方法从而sleep 6秒，让客户端(nginx or ribbon)有足够的时间检测到本服务已经下线从而实现优雅停机
     * （客户端会每隔3秒(ribbon是3秒，nginx是2秒)发送一次心跳，如果/monitor/check接口已经下线则移除对该服务器的访问）
     */
    public static class GracefulCloseLifeCycle implements SmartLifecycle{

        private volatile boolean isRunning = false;
        private final int sleepSeconds;

        public GracefulCloseLifeCycle(int sleepSeconds){
            this.sleepSeconds = sleepSeconds;
        }

        @Override
        public boolean isAutoStartup() {
            return true;
        }

        @Override
        public void stop(final Runnable callback) {
            isRunning = false;
            new Thread(){
                @Override
                public void run() {
                    try{
                        if(!logger.isInfoEnabled()) {
                            LoggerContext loggerContext = (LoggerContext) LoggerFactory.getILoggerFactory();
                            loggerContext.getLogger(GracefulCloseAutoConfiguration.class).setLevel(Level.INFO);
                        }
                        //在调用到这里之前，GracefulCloseRunListener已经把/monitor/check接口置为返回FAIL
                        logger.info("sleep {} seconds to wait nginx/ribbon check", sleepSeconds);
                        TimeUnit.SECONDS.sleep(sleepSeconds);
                    }catch(InterruptedException e){
                        //Ignore
                    }finally {
                        callback.run();
                    }
                }
            }.start();
        }

        @Override
        public void start() {
            //do nothing
            isRunning = true;
        }

        @Override
        public void stop() {
            //will not be invoked
        }

        @Override
        public boolean isRunning() {
            return isRunning;
        }

        /**
         * Eureka/Consul的LifeCycle的phase是0，所以这里设置为-1，让本类的stop会在取消注册之后被调用
         * @return
         */
        @Override
        public int getPhase() {
            return -1;
        }
    }
}
