package cn.com.duibaboot.ext.autoconfigure.core;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.BeanCreationNotAllowedException;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.event.SimpleApplicationEventMulticaster;
import org.springframework.context.support.AbstractApplicationContext;

/**
 * 配置自定义的ApplicationEventMulticaster广播器，加入ErrorHandler错误处理器，以修复部分spring的bug。
 * 启动时这个广播器没有配置线程池，事件仍然按顺序串行处理，不影响原先流程，此时错误处理器不会生效。
 * 在spring容器关闭前，GracefulCloseRunListener会给这个广播器加入线程池，让所有事件并行处理（关闭时事件处理顺序无所谓），并让错误处理器生效，从而有机会拦截spring的一个异常（spring的bug）。
 */
@Configuration
@ConditionalOnProperty(name="spring.event.use-executor", havingValue = "true", matchIfMissing = true)
public class ApplicationEventMulticasterAutoConfiguration {

    private static final Logger logger = LoggerFactory.getLogger(ApplicationEventMulticasterAutoConfiguration.class);

    @Bean(name= AbstractApplicationContext.APPLICATION_EVENT_MULTICASTER_BEAN_NAME)
    public SimpleApplicationEventMulticaster simpleApplicationEventMulticaster(){
        SimpleApplicationEventMulticaster multicaster = new SimpleApplicationEventMulticaster();
        multicaster.setErrorHandler(t -> {
            if(t instanceof BeanCreationNotAllowedException){
                //修复spring的bug，忽略此异常；这个异常会在关闭时报出，原因是主容器销毁时，先销毁了eurekaAutoServiceRegistration这个bean，然后开始销毁feign子容器，子容器销毁时又发出了ContextClosedEvent, 这个事件会传播到主容器，主容器还保留着对eurekaAutoServiceRegistration里listener的引用，调用listener时尝试获取eurekaAutoServiceRegistration，而这个bean已经销毁，会尝试重建，而容器在销毁时不允许新建bean，所以报错
                if("eurekaAutoServiceRegistration".equals(((BeanCreationNotAllowedException)t).getBeanName())
                        && t.getMessage() != null
                        && t.getMessage().contains("Singleton bean creation not allowed while singletons of this factory are in destruction")){
                    return;
                }
            }
            logger.warn("Error calling ApplicationEventListener", t);
        });
        //这里不能设置线程池，GracefulCloseRunListener中会在spring关闭前设置线程池
//        multicaster.setTaskExecutor(TtlExecutors.getTtlExecutorService(new ThreadPoolExecutor(1, 5, 10, TimeUnit.SECONDS, new LinkedBlockingQueue<>())));
        return multicaster;
    }
}
