package cn.com.duibaboot.ext.autoconfigure.config;

import com.google.common.base.Joiner;
import org.apache.commons.lang3.StringUtils;
import org.springframework.boot.SpringApplication;
import org.springframework.boot.autoconfigure.data.redis.RedisAutoConfiguration;
import org.springframework.boot.env.EnvironmentPostProcessor;
import org.springframework.core.env.ConfigurableEnvironment;
import org.springframework.core.env.MapPropertySource;
import org.springframework.core.env.PropertySource;

import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

/**
 * 添加一些配置以禁止某些无用特性,比如servo等.<br/>
 * 配置undertow使用tomcat的配置, 如果没有，则设置undertow线程池大小为200。
 */
public class DuibaEnvironmentPostProcessor implements EnvironmentPostProcessor {
    private static final Map<String, Object> defProperties;
    static {
        Map<String, Object> overrideProperties = new HashMap<>();
        //配置禁用netflix servo监控
        overrideProperties.put("spring.metrics.servo.enabled", "false");
        overrideProperties.put("spring.metrics.export.enabled", "false");

        defProperties = Collections.unmodifiableMap(overrideProperties);
    }

    @Override
    public void postProcessEnvironment(ConfigurableEnvironment environment, SpringApplication application) {
        if(!application.isWebEnvironment()){//判断到是bootstrap的application则跳过
            return;
        }
        Map<String, Object> properties = new HashMap<>();
        properties.putAll(defProperties);

        PropertySource<?> propertySource = new MapPropertySource("duibaBootDefaultConfig",
                properties);
        //最后的优先级最低
        environment.getPropertySources().addLast(propertySource);

        configEmbedServletContainerProperties(environment, properties);
        configRedisProperties(environment, properties);
    }

    private void configEmbedServletContainerProperties(ConfigurableEnvironment environment, Map<String, Object> properties){
        String undertowWorkerThreads = environment.getProperty("server.undertow.workerThreads");
        if(StringUtils.isBlank(undertowWorkerThreads)){
            undertowWorkerThreads = environment.getProperty("server.undertow.worker-threads");
        }

        if(!StringUtils.isBlank(undertowWorkerThreads)){
            return;
        }

        String tomcatWorkerThreads = environment.getProperty("server.tomcat.maxThreads");
        if(StringUtils.isBlank(undertowWorkerThreads)){
            tomcatWorkerThreads = environment.getProperty("server.tomcat.max-threads");
        }

        if(!StringUtils.isBlank(tomcatWorkerThreads)){//优先使用tomcat的线程数来赋值给undertow
            properties.put("server.undertow.worker-threads", tomcatWorkerThreads);
        } else {//否则配置使用默认值200
            properties.put("server.undertow.worker-threads", "200");
        }
    }

    private void configRedisProperties(ConfigurableEnvironment environment, Map<String, Object> properties){
        //禁用Spring官方的 RedisAutoConfiguration。
        String property = "spring.autoconfigure.exclude";
        String excludeAutoConfigClasses = StringUtils.defaultString(environment.getProperty(property), null);
        excludeAutoConfigClasses = Joiner.on(",").skipNulls().join(excludeAutoConfigClasses, RedisAutoConfiguration.class.getName());
        properties.put(property, excludeAutoConfigClasses);

        if (StringUtils.isBlank(environment.getProperty("duiba.redis.host"))) {
            //禁用 RedisRepositoriesAutoConfiguration
            properties.put("spring.data.redis.repositories.enabled", "false");
        }
    }
}
