package cn.com.duibaboot.ext.autoconfigure.rocketmq;

import cn.com.duibaboot.ext.autoconfigure.core.EarlyClose;
import org.apache.rocketmq.client.consumer.DefaultMQPushConsumer;
import org.apache.rocketmq.client.consumer.listener.*;
import org.apache.rocketmq.client.exception.MQClientException;
import org.apache.rocketmq.client.producer.DefaultMQProducer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.FactoryBean;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.NoSuchBeanDefinitionException;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.ApplicationContext;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.util.ReflectionUtils;

import javax.annotation.Resource;
import java.lang.reflect.Field;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

/**
 * RocketMQ生产者/消费者自动配置
 * Created by wenqi.huang on 2017/04/18.
 */
@Configuration
@EnableConfigurationProperties(RocketMqProperties.class)
public class RocketMqAutoConfiguration {
    private static final Logger logger = LoggerFactory.getLogger(RocketMqAutoConfiguration.class);

    protected static abstract class RocketMqClientConfiguration{

        @Autowired
        protected RocketMqProperties rocketMqProperties;

    }

    @Configuration
    @ConditionalOnClass({DefaultMQProducer.class})
    @ConditionalOnMissingBean(name={"rocketMqProducer"})
    @ConditionalOnProperty(name="duiba.rocketmq.producer.enable", havingValue = "true", matchIfMissing = false)
    protected static class RocketMqProducerConfigurator extends RocketMqClientConfiguration{

        @Bean(name="rocketMqProducer", destroyMethod = "shutdown")
        public DefaultMQProducer rocketMqProducer() throws MQClientException {
            /**
             * 一个应用创建一个Producer，由应用来维护此对象，可以设置为全局对象或者单例<br>
             */
            DefaultMQProducerWrapper p = new DefaultMQProducerWrapper();

            /**
             * producerGroup 这个概念发送普通的消息时，作用不大，但是发送分布式事务消息时，比较关键，
             * 因为服务器会回查这个Group下的任意一个Producer
             * 建议把producerGroup设置成应用名
             */
            p.setProducerGroup(rocketMqProperties.getProducer().getGroup());
            p.setNamesrvAddr(rocketMqProperties.getNameSrvAddr());
            //p.setInstanceName("Producer");
            p.setSendMsgTimeout(rocketMqProperties.getProducer().getSendMsgTimeoutMillis());//单位：ms
            /**
             * Producer对象在使用之前必须要调用start初始化，初始化一次即可<br>
             * 注意：切记不可以在每次发送消息时，都调用start方法
             */
            p.start();

            return p;
        }
    }

    @Configuration
    @ConditionalOnClass({DefaultMQPushConsumer.class})
    protected static class RocketMqConsumerConfigurator extends RocketMqClientConfiguration{

        @Bean(name="rocketMqConsumer")
        @ConditionalOnMissingBean(name={"rocketMqConsumer"})
        @ConditionalOnProperty(name="duiba.rocketmq.consumer.enable", havingValue = "true", matchIfMissing = false)
        public FactoryBean<DefaultMQPushConsumer> rocketMqConsumer() throws ClassNotFoundException, IllegalAccessException, InstantiationException {
            return new RocketMqConsumerFactoryBean(rocketMqProperties, -1);
        }

        @Bean(name="extraRocketMqConsumer0")
        @ConditionalOnMissingBean(name={"extraRocketMqConsumer0"})
        @ConditionalOnProperty(name="duiba.rocketmq.extra-consumer[0].enable", havingValue = "true", matchIfMissing = false)
        public FactoryBean<DefaultMQPushConsumer> extraRocketMqConsumer0() throws ClassNotFoundException, IllegalAccessException, InstantiationException {
            return new RocketMqConsumerFactoryBean(rocketMqProperties, 0);
        }

        @Bean(name="extraRocketMqConsumer1")
        @ConditionalOnMissingBean(name={"extraRocketMqConsumer1"})
        @ConditionalOnProperty(name="duiba.rocketmq.extra-consumer[1].enable", havingValue = "true", matchIfMissing = false)
        public FactoryBean<DefaultMQPushConsumer> extraRocketMqConsumer1() throws ClassNotFoundException, IllegalAccessException, InstantiationException {
            return new RocketMqConsumerFactoryBean(rocketMqProperties, 1);
        }

        @Bean(name="extraRocketMqConsumer2")
        @ConditionalOnMissingBean(name={"extraRocketMqConsumer2"})
        @ConditionalOnProperty(name="duiba.rocketmq.extra-consumer[2].enable", havingValue = "true", matchIfMissing = false)
        public FactoryBean<DefaultMQPushConsumer> extraRocketMqConsumer2() throws ClassNotFoundException, IllegalAccessException, InstantiationException {
            return new RocketMqConsumerFactoryBean(rocketMqProperties, 2);
        }

    }

    private static class RocketMqConsumerFactoryBean extends EarlyClose implements FactoryBean<DefaultMQPushConsumer>,InitializingBean{

        @Resource
        private ApplicationContext applicationContext;

        private DefaultMQPushConsumer rocketMqConsumer;
        private RocketMqProperties rocketMqProperties;
        private RocketMqProperties.ConsumerProperties consumerProperties;
        private String messageListenerBeanId;

        public RocketMqConsumerFactoryBean(RocketMqProperties rocketMqProperties, int consumerPropertiesIndex){
            this.rocketMqProperties = rocketMqProperties;
            if(consumerPropertiesIndex == -1){//默认consumer
                consumerProperties = rocketMqProperties.getConsumer();
                messageListenerBeanId = "bootRocketMqMessageListener";
            }else{//extra-consumer
                consumerProperties = rocketMqProperties.getExtraConsumer()[consumerPropertiesIndex];
                messageListenerBeanId = "extraBootRocketMqMessageListener" + consumerPropertiesIndex;
            }
        }

        @Override
        public void afterPropertiesSet() throws Exception {
            MessageListener rocketMqMessageListener;

            try {
                rocketMqMessageListener = applicationContext.getBean(messageListenerBeanId, MessageListener.class);
            }catch(NoSuchBeanDefinitionException e){
                logger.error("", new IllegalStateException("bean id:[" + messageListenerBeanId + "] class:[org.apache.rocketmq.client.consumer.listener.MessageListenerConcurrently] is not exist in spring's context, 请声明，否则不会启用rocketmq消费!!!，请参考: http://gitlab2.dui88.com/basic_platform/spring-boot-ext/wikis/Starter%E7%89%B9%E6%80%A7/_rocketmq%E3%80%90%E5%BC%80%E6%BA%90RocketMQ%E3%80%91"));
                return;
            }

            /**
             * 一个应用创建一个Consumer，由应用来维护此对象，可以设置为全局对象或者单例<br>
             * 注意：consumerGroup 需要由应用来保证唯一
             * consumerGroup建议使用应用名。RocketMQ使用consumerGroup作为负载均衡的粒度，
             * 所以同一个consumerGroup订阅的Topic必须完全相同。
             * 考虑这样的场景，活动中心之前已经订阅了一些topic，某次发布新版需要增加一个topic的订阅，发布一部分服务器的情况下，新的topic的某些queue不会被消费，所以需要确保活动中心全发。某个版本如果新增了topic，则不能做灰度发布。考虑到代码可读性和性能，这个代价是值得的
             */
            //DefaultMQPullConsumer consumer = new DefaultMQPullConsumer();
            DefaultMQPushConsumer consumer = new DefaultMQPushConsumer(consumerProperties.getGroup());
            //consumer.setConsumerGroup();
            consumer.setNamesrvAddr(rocketMqProperties.getNameSrvAddr());
            consumer.setMaxReconsumeTimes(consumerProperties.getMaxReconsumeTimes());//最大重复消费次数，超过这个次数还没被确认则丢弃消息，请根据消息重要程度设置
            if(consumerProperties.getConsumeThreadNums() > 0) {
                consumer.setConsumeThreadMin(consumerProperties.getConsumeThreadNums());//消费最小线程数
                consumer.setConsumeThreadMax(Math.max(consumerProperties.getConsumeThreadNums(), 64));//消费最大线程数
            }
            //指定集群消费模型/广播消费模型
            consumer.setMessageModel(consumerProperties.getMessageModelEnum());
            //consumer.setConsumeTimeout(15);
            //consumer.setPullBatchSize(32);
            consumer.setConsumeMessageBatchMaxSize(1);//监听器每次接受本地队列的消息是多少条(从服务器获取还是多条的，只是本地MessageListener每次处理只有一条)
            //consumer.setMessageModel(MessageModel.BROADCASTING);
            //consumer.setPullInterval();
            //consumer.setInstanceName("Consumber");

            String topics = consumerProperties.getTopics();
            String[] topicArr = topics.split(",");

            for(String topic : topicArr) {
                consumer.subscribe(topic, "*");
            }
            //尽量使用MessageListenerConcurrently来消费，吞吐量更高，消费无顺序
            //当对消息消费顺序有严格要求时，请使用MessageListenerOrderly
            if(rocketMqMessageListener instanceof MessageListenerConcurrently) {
                consumer.registerMessageListener((MessageListenerConcurrently)rocketMqMessageListener);
            }else if(rocketMqMessageListener instanceof MessageListenerOrderly) {
                consumer.registerMessageListener((MessageListenerOrderly)rocketMqMessageListener);
            }

            /**
             * Consumer对象在使用之前必须要调用start初始化，初始化一次即可<br>
             */
            consumer.start();

            rocketMqConsumer = consumer;
        }

        @Override
        public DefaultMQPushConsumer getObject() throws Exception {
            return rocketMqConsumer;
        }

        @Override
        public Class<?> getObjectType() {
            return DefaultMQPushConsumer.class;
        }

        @Override
        public boolean isSingleton() {
            return true;
        }

        @Override
        public void stop() {
            if(rocketMqConsumer != null) {
                rocketMqConsumer.shutdown();
                awaitShutdown(rocketMqConsumer, 3);
            }
        }

        /**
         * 通过反射取出主线程池并等待一会,优雅停机
         * @param defaultMQPushConsumer
         * @param awaitMaxSeconds
         */
        private void awaitShutdown(DefaultMQPushConsumer defaultMQPushConsumer, int awaitMaxSeconds){
            Field fieldDefaultMQPushConsumerImpl = ReflectionUtils.findField(defaultMQPushConsumer.getClass(), "defaultMQPushConsumerImpl");
            fieldDefaultMQPushConsumerImpl.setAccessible(true);
            Object defaultMQPushConsumerImpl = ReflectionUtils.getField(fieldDefaultMQPushConsumerImpl, defaultMQPushConsumer);

            Field fieldConsumeMessageService = ReflectionUtils.findField(defaultMQPushConsumerImpl.getClass(), "consumeMessageService");
            fieldConsumeMessageService.setAccessible(true);
            Object consumeMessageService = ReflectionUtils.getField(fieldConsumeMessageService, defaultMQPushConsumerImpl);

            Field fieldConsumeExecutor = ReflectionUtils.findField(consumeMessageService.getClass(), "consumeExecutor");
            fieldConsumeExecutor.setAccessible(true);
            ThreadPoolExecutor consumeExecutor = (ThreadPoolExecutor)ReflectionUtils.getField(fieldConsumeExecutor, consumeMessageService);

            try {
                consumeExecutor.awaitTermination(awaitMaxSeconds, TimeUnit.SECONDS);
            } catch (InterruptedException e) {
                //Ignore
            }
        }

        //和onsConsumer一起关闭，且在统一线程池之前关闭
        @Override
        public int getPhase() {
            return 1;
        }
    }
}
