package cn.com.duibaboot.ext.autoconfigure.javaagent.core;

import cn.com.duibaboot.ext.autoconfigure.javaagent.core.interceptor.enhance.ClassEnhancePluginDefine;
import cn.com.duibaboot.ext.autoconfigure.javaagent.core.match.ClassMatch;
import net.bytebuddy.dynamic.DynamicType;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Basic abstract class of all agent auto-instrumentation plugins.
 * <p>
 * It provides the outline of enhancing the target class.
 * If you want to know more about enhancing, you should go to see {@link ClassEnhancePluginDefine}
 */
public abstract class AbstractClassEnhancePluginDefine {
    private final Logger logger =  LoggerFactory.getLogger(AbstractClassEnhancePluginDefine.class.getName());

    /**
     * Main entrance of enhancing the class.
     *
     * @param transformClassName target class.
     * @param builder byte-buddy's builder to manipulate target class's bytecode.
     * @param classLoader load the given transformClass
     * @return the new builder, or <code>null</code> if not be enhanced.
     * @throws PluginException, when set builder failure.
     */
    public DynamicType.Builder<?> define(String transformClassName,
        DynamicType.Builder<?> builder, ClassLoader classLoader) throws PluginException {
        String interceptorDefineClassName = this.getClass().getName();

        if (StringUtils.isEmpty(transformClassName)) {
            if(logger.isWarnEnabled()) {
                logger.warn("classname of being intercepted is not defined by " + interceptorDefineClassName);
            }
            return null;
        }

        if(logger.isInfoEnabled()) {
            logger.info("prepare to enhance class " + transformClassName + " by "+interceptorDefineClassName);
        }

        /**
         * find witness classes for enhance class
         */
        String[] witnessClasses = witnessClasses();
        if (witnessClasses != null) {
            for (String witnessClass : witnessClasses) {
                if (!WitnessClassFinder.INSTANCE.exist(witnessClass, classLoader)) {
                    if(logger.isWarnEnabled()) {
                        logger.warn("enhance class "+transformClassName+" by plugin "+interceptorDefineClassName+" is not working. Because witness class "+witnessClass+" is not existed.");
                    }
                    return null;
                }
            }
        }

        /**
         * find origin class source code for interceptor
         */
        DynamicType.Builder<?> newClassBuilder = this.enhance(transformClassName, builder, classLoader);

        if(logger.isInfoEnabled()) {
            logger.info("enhance class "+transformClassName+" by "+interceptorDefineClassName+" completely.");
        }

        return newClassBuilder;
    }

    protected abstract DynamicType.Builder<?> enhance(String enhanceOriginClassName,
        DynamicType.Builder<?> newClassBuilder, ClassLoader classLoader) throws PluginException;

    /**
     * Define the {@link ClassMatch} for filtering class.
     *
     * @return {@link ClassMatch}
     */
    protected abstract ClassMatch enhanceClass();

    /**
     * Witness classname list. Why need witness classname? Let's see like this: A library existed two released versions
     * (like 1.0, 2.0), which include the same target classes, but because of version iterator, they may have the same
     * name, but different methods, or different method arguments list. So, if I want to target the particular version
     * (let's say 1.0 for example), version number is obvious not an option, this is the moment you need "Witness
     * classes". You can add any classes only in this particular release version ( something like class
     * com.company.1.x.A, only in 1.0 ), and you can achieve the goal.
     *
     * @return
     */
    protected String[] witnessClasses() {
        return new String[] {};
    }
}
