package org.springframework.cloud.netflix.feign;

import cn.com.duiba.boot.cat.CatWithArgs;
import cn.com.duiba.boot.exception.BizException;
import cn.com.duibaboot.ext.autoconfigure.core.rpc.RpcContext;
import cn.com.duibaboot.ext.autoconfigure.core.utils.CatUtils;
import cn.com.duiba.catmonitor.constants.CatConstants;
import com.dianping.cat.Cat;
import com.dianping.cat.message.Event;
import com.dianping.cat.message.Message;
import com.dianping.cat.message.Transaction;
import com.dianping.cat.message.internal.AbstractMessage;
import com.google.common.annotations.VisibleForTesting;
import com.netflix.hystrix.exception.HystrixBadRequestException;
import org.aopalliance.intercept.MethodInterceptor;

import java.lang.reflect.InvocationHandler;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.lang.reflect.Proxy;
import java.util.HashMap;
import java.util.Map;


/**
 * 拦截FeignClient的调用加入cat监控功能，以及拦截并注入当前调用的方法
 */
public class CustomFeignClientFactoryBean extends FeignClientFactoryBean {

    //客户端应用名，CustomFeignClientsRegistrar.customFeignClientBeanDefinition会注入
//    @Value("${spring.application.name}")
    private String springApplicationName;

    @Override
    public Object getObject() throws Exception {
        //增加cat监控功能
        Class classType = super.getObjectType();
        Object feignClient = super.getObject();

        return Proxy.newProxyInstance(classType.getClassLoader(), new Class[]{classType}, new FeignClientInvocationHandler(springApplicationName, feignClient, this.getName()));
    }

    public void setSpringApplicationName(String springApplicationName) {
        this.springApplicationName = springApplicationName;
    }


    /**
     * {@link MethodInterceptor}
     */
    @VisibleForTesting
    public static class FeignClientInvocationHandler implements InvocationHandler {

        private String springApplicationName;
        private String serverName;
        private Object original;

        FeignClientInvocationHandler(String springApplicationName, Object original, String serverName) {
            this.springApplicationName = springApplicationName;
            this.original = original;
            this.serverName = serverName;
        }

        @Override
        public Object invoke(Object proxy, Method method, Object[] args) throws Throwable {
            String methodName = method.getName();

            if ("equals".equals(methodName)
                    || "hashCode".equals(methodName)
                    || "toString".equals(methodName)
                    || method.isDefault()) {
                return invokeMethodInner(method, original, args);
            }

            RpcContext.getContext().setMethod(method);
            RpcContext.getContext().setInvokeArgs(args);

            if(!CatUtils.isCatEnabled()){
                return invokeMethodInner(method, original, args);
            }

            String loggerName = resolveLoggerName(method, args);

            Transaction transaction = Cat.newTransaction(CatConstants.CROSS_CONSUMER, loggerName);

            try {
                Cat.Context context = new RpcCatContext();

                Cat.logRemoteCallClient(context);
                setAttachment(context, method);

                Object result = invokeMethodInner(method, original, args);

                transaction.setStatus(Message.SUCCESS);
                return result;
            }catch(Throwable e){
                if(!(e instanceof BizException)) {
                    Cat.logError(e);//BizException是正常的业务异常，比如用户不存在，不应该记录到cat.
                    transaction.setStatus(e);
                }else{
                    transaction.setStatus(String.format("BizException(message:%s,code:%s)(this exception will not trigger hystrix)",e.getMessage(),((BizException) e).getCode()));
                }
                throw e;
            }finally {
                createConsumerCross(transaction);
                CatUtils.completeTransaction(transaction);
            }
        }

        private String resolveLoggerName(Method method, Object[] args){
            StringBuilder loggerName = new StringBuilder(method.getDeclaringClass().getSimpleName())
                    .append(".").append(method.getName());

            //注解了CatWithArgs表示cat中的监控名需要附加参数，详情见CatWithArgs类的注释
            if(method.isAnnotationPresent(CatWithArgs.class)){
                CatWithArgs catWithArgs = method.getAnnotation(CatWithArgs.class);
                int[] argIndexes = catWithArgs.argIndexes();
                if(argIndexes == null || argIndexes.length == 0
                        || args == null || args.length == 0){
                    return loggerName.toString();
                }

                StringBuilder sb = new StringBuilder();
                for(int argIdx : argIndexes){
                    if(argIdx >= 0 && args.length > argIdx) {
                        Object obj = args[argIdx];
                        sb.append(obj == null ? "" : obj.toString()).append(",");
                    }
                }

                if(sb.length() > 0){
                    sb.deleteCharAt(sb.length() - 1);
                    sb.insert(0, "(");
                    sb.append(")");
                }
                loggerName.append(sb);
            }
            return loggerName.toString();
        }

        private Object invokeMethodInner(Method method, Object target, Object[] args) throws Throwable{
            try {
                return method.invoke(target, args);
            } catch (Throwable e) {
                if(e instanceof InvocationTargetException){
                    e = ((InvocationTargetException)e).getTargetException();
                }
                //通常这步拿出来的异常只会有以下2种：HystrixBadRequestException, FeignClientException
                if(e instanceof HystrixBadRequestException && e.getCause() != null && e.getCause() instanceof BizException){
                    e = e.getCause();//如果cause是BizException，则应该把BizException提取出来抛出
                }
                throw e;
            }
        }

        private static class RpcCatContext implements Cat.Context{

            private Map<String,String> properties = new HashMap<String, String>();

            @Override
            public void addProperty(String key, String value) {
                properties.put(key,value);
            }

            @Override
            public String getProperty(String key) {
                return properties.get(key);
            }
        }

        private void setAttachment(Cat.Context context, Method method){
            RpcContext.getContext().setAttachment(Cat.Context.ROOT,context.getProperty(Cat.Context.ROOT));
            RpcContext.getContext().setAttachment(Cat.Context.CHILD,context.getProperty(Cat.Context.CHILD));
            RpcContext.getContext().setAttachment(Cat.Context.PARENT,context.getProperty(Cat.Context.PARENT));
            RpcContext.getContext().setAttachment(cn.com.duibaboot.ext.autoconfigure.core.Constants.X_RPC_CLIENT, springApplicationName);
        }

        private void createConsumerCross(Transaction transaction){
            Event crossAppEvent =  Cat.newEvent(CatConstants.CONSUMER_CALL_APP, springApplicationName + "[" + RpcContext.getContext().getLocalAddr() + "]");
            Event crossServerEvent = Cat.newEvent(CatConstants.CONSUMER_CALL_SERVER, serverName + "[" + RpcContext.getContext().getRemoteAddr() + "]");
//            Event crossPortEvent = Cat.newEvent(CatConstants.CONSUMER_CALL_PORT,url.getPort()+"");
//            completeEvent(crossAppEvent);
//            completeEvent(crossPortEvent);
//            completeEvent(crossServerEvent);

            transaction.addChild(crossAppEvent);
//            transaction.addChild(crossPortEvent);
            transaction.addChild(crossServerEvent);
        }

        private void completeEvent(Event event){
            AbstractMessage message = (AbstractMessage) event;
            message.setCompleted(true);
            message.setStatus(Event.SUCCESS);
        }

    }

}
