package cn.com.duibaboot.ext.autoconfigure.perftest;

import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import javax.sql.DataSource;

import cn.com.duiba.boot.perftest.PerfTestContext;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import org.apache.commons.dbcp2.BasicDataSource;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.cloud.context.environment.EnvironmentChangeEvent;
import org.springframework.context.event.EventListener;
import org.springframework.core.env.AbstractEnvironment;
import org.springframework.core.env.CompositePropertySource;
import org.springframework.core.env.EnumerablePropertySource;
import org.springframework.core.env.Environment;
import org.springframework.core.env.MapPropertySource;
import org.springframework.core.env.PropertiesPropertySource;
import org.springframework.core.env.PropertySource;
import org.springframework.jdbc.datasource.lookup.AbstractRoutingDataSource;
import org.springframework.util.Assert;

/**
 * 自动根据当前是不是压测请求决定访问原始数据库或者影子库
 * Created by wenqi.huang on 2016/11/24.
 */
public class PerfTestRoutingDataSource extends AbstractRoutingDataSource implements DisposableBean {

    /**
     * 原始库的key
     */
    private static final String ORIGINAL_DATASOURCE = "original";
    /**
     * 影子库的key
     */
    private static final String TEST_DATASOURCE = "test";
    /**
     * 影子库的scheme前缀
     */
    protected static final String TEST_SCHEME_PREFIX = "perf__";
    private static final String SHADE_SCHEMA = ".shade";

    private volatile boolean hasTestDataSource = false;
    private volatile boolean testDataSourceInitted = false;
    /**
     * 原始库的sheme
     */
    private String originalScheme;

    /**
     * 原始库datasource
     */
    private BasicDataSource originalDataSource;
    private Environment environment;
    private BasicDataSource shadeDataSource;
    private ConcurrentMap<Object, Object> targetDataSources = new ConcurrentHashMap<>();
    private boolean thisInitted = false;
    private Map<String, String> dbShade = Maps.newHashMap();
    private List<String> dbShadeKeys = Lists.newArrayList();
    public PerfTestRoutingDataSource(DataSource originalDataSource, Environment environment) {
        if(originalDataSource instanceof BasicDataSource){//dbcp2
            this.originalDataSource = (BasicDataSource)originalDataSource;
            this.environment = environment;
        }else{
            throw new IllegalStateException("检测到有DataSource没有使用dbcp2，请先改成dbcp2。");
            //logger.error("检测到有DataSource没有使用dbcp2，请先改成dbcp2。");
        }
    }

    @Override
    public void afterPropertiesSet() {
        if(thisInitted){
            return;
        }
        thisInitted = true;

        targetDataSources.put(ORIGINAL_DATASOURCE, originalDataSource);
        String originalUrl = (originalDataSource).getUrl();
        originalScheme = getOriginalScheme(originalUrl);

        super.setTargetDataSources(targetDataSources);
        initShadeDbMap();
        super.afterPropertiesSet();
    }

    private void initShadeDbMap() {
        Properties properties = fetchAllProperties();
        properties.entrySet().stream().forEach(x -> {
            if (x.getKey().toString().endsWith(SHADE_SCHEMA)) {
                String dbUrl = environment.getProperty(x.getKey().toString().replace(SHADE_SCHEMA, ""));
                if (originalDataSource.getUrl().equals(dbUrl)) {//是当前数据源的影子配置时才加入
                    dbShade.put(dbUrl, x.getValue().toString());
                    if (!dbShadeKeys.contains(x.getKey().toString())) {
                        dbShadeKeys.add(x.getKey().toString());
                    }
                }
            }
        });
    }

    /**
     * 根据原始数据源得到对应的影子DataSource，影子数据库不存在则返回null.
     * 按照约定，影子库url和原始库url一致，只是scheme部分加入前缀perf__,用户名密码也必须相同，同一个用户要能看到这两个库
     * @param originalDataSource
     */
    private BasicDataSource getTestDataSource(BasicDataSource originalDataSource){
        Connection connection = null;
        Statement statement = null;
        ResultSet rs = null;
        Set<String> schemes = new HashSet<>();
        //用show databases命令检测所有数据库scheme，目前只支持mysql数据库的检测
        try {
            connection = originalDataSource.getConnection();
            statement = connection.createStatement();
            rs = statement.executeQuery("show databases");
            while (rs.next()) {
                schemes.add(rs.getString(1));
            }
        } catch (SQLException e) {
            throw new RuntimeException(e);
        } finally {
            if(rs != null) {
                try {
                    rs.close();
                }catch(Exception e){
                    // ignore
                }
            }
            if(statement != null) {
                try {
                    statement.close();
                }catch(Exception e){
                    // ignore
                }
            }
            if(connection != null) {
                try {
                    connection.close();
                }catch(Exception e){
                    // ignore
                }
            }
        }

        String testScheme = TEST_SCHEME_PREFIX + originalScheme;
        if(!schemes.contains(testScheme)&&!dbShade.containsKey(originalDataSource.getUrl())){//影子库不存在
            return null;
        }

        String testUrl = getTestUrl(originalDataSource.getUrl());

        BasicDataSource ds = new BasicDataSource();
        ds.setDriverClassName(originalDataSource.getDriverClassName());
        ds.setUrl(testUrl);
        ds.setUsername(originalDataSource.getUsername());
        ds.setPassword(originalDataSource.getPassword());
        ds.setDefaultQueryTimeout(originalDataSource.getDefaultQueryTimeout());
        ds.setInitialSize(originalDataSource.getInitialSize());
        ds.setLogAbandoned(originalDataSource.getLogAbandoned());
        ds.setMaxIdle(originalDataSource.getMaxIdle());
        ds.setMaxTotal(originalDataSource.getMaxTotal());
        ds.setMinIdle(originalDataSource.getMinIdle());
        ds.setSoftMinEvictableIdleTimeMillis(originalDataSource.getSoftMinEvictableIdleTimeMillis());
        ds.setValidationQuery(originalDataSource.getValidationQuery());
        ds.setValidationQueryTimeout(originalDataSource.getValidationQueryTimeout());
        ds.setTimeBetweenEvictionRunsMillis(originalDataSource.getTimeBetweenEvictionRunsMillis());
        ds.setTestWhileIdle(originalDataSource.getTestWhileIdle());
        ds.setTestOnBorrow(originalDataSource.getTestOnBorrow());
        ds.setTestOnReturn(originalDataSource.getTestOnReturn());
        ds.setTestOnCreate(originalDataSource.getTestOnCreate());
        ds.setMaxWaitMillis(originalDataSource.getMaxWaitMillis());
        ds.setNumTestsPerEvictionRun(originalDataSource.getNumTestsPerEvictionRun());
        ds.setRemoveAbandonedOnBorrow(originalDataSource.getRemoveAbandonedOnBorrow());
        ds.setRemoveAbandonedOnMaintenance(originalDataSource.getRemoveAbandonedOnMaintenance());
        ds.setRemoveAbandonedTimeout(originalDataSource.getRemoveAbandonedTimeout());

        return ds;
    }

    /**
     * 根据原始的数据源url得到对应的影子库url，按照约定，影子库url和原始库url一致，只是scheme部分加入前缀perf__
     * @param originalUrl
     * @return
     */
    private String getTestUrl(String originalUrl){

        if (dbShade.containsKey(originalUrl)) {
            return dbShade.get(originalUrl);//如果存在影子库单独配置则使用改配置
        }
        int index = originalUrl.lastIndexOf("/");
        if(index > 0){
            return originalUrl.substring(0, index + 1) + TEST_SCHEME_PREFIX + originalUrl.substring(index+1, originalUrl.length());
        }else{
            throw new IllegalArgumentException(originalUrl + " is invalid");
        }
    }

    /**
     * 根据原始的数据源url得到它的scheme
     * @param originalUrl
     * @return
     */
    private String getOriginalScheme(String originalUrl){
        int index = originalUrl.lastIndexOf("/");
        if(index > 0){
            String schemeAfter = originalUrl.substring(index+1, originalUrl.length());

            int index1 = schemeAfter.indexOf("?");
            if(index1 > 0){
                return schemeAfter.substring(0, index1);
            }else{
                return schemeAfter;
            }
        }else{
            throw new IllegalArgumentException(originalUrl + " is invalid");
        }
    }

    protected DataSource determineTargetDataSource() {
        Assert.notNull(this.targetDataSources, "DataSource router not initialized");
        Object lookupKey = determineCurrentLookupKey();
        DataSource dataSource = (DataSource)this.targetDataSources.get(lookupKey);

        if (dataSource == null) {
            throw new IllegalStateException("Cannot determine target DataSource for lookup key [" + lookupKey + "]");
        }
        return dataSource;
    }

    /**
     * 调用refresh后配置变更,重新初始化测试库
     */
    @EventListener(EnvironmentChangeEvent.class)
    public void onEnvironmentChange(EnvironmentChangeEvent event) {
        if (dbShadeKeys.stream().filter(x -> event.getKeys().contains(x)).count() > 0) {
            initShadeDbMap();
            if (testDataSourceInitted) {
                testDataSourceInitted = false;
            }
        }
    }

    protected Object determineCurrentLookupKey(){
        if(PerfTestContext.isCurrentInPerfTestMode()){//压测时，就算压测影子库不存在也不走原始库，防止产生脏数据
            if(!testDataSourceInitted){//在这里延迟加载
                initTestDataSource();
            }

            if(!hasTestDataSource){
                throw new IllegalStateException(originalScheme + "的影子库不存在，放弃本次请求");
            }
            PerfTestContext.debugInfo("PerfTest DB");
            return TEST_DATASOURCE;
        }

        return ORIGINAL_DATASOURCE;
    }

    /**
     * 初始化影子库的DataSource，如果找不到影子库，则为null
     */
    private void initTestDataSource(){
        synchronized (this){
            if(!testDataSourceInitted) {
                BasicDataSource testDataSource = getTestDataSource(originalDataSource);

                if (testDataSource != null) {
                    hasTestDataSource = true;
                    shadeDataSource = testDataSource;
                    targetDataSources.put(TEST_DATASOURCE, testDataSource);//替换
                } else {
                    logger.warn(originalScheme + " 没有对应的影子数据库（在同个库中以perf__开头的scheme），如果你配置了，需要重启本应用生效");
                }

                testDataSourceInitted = true;
            }
        }
    }

    @Override
    public void destroy() throws Exception {
        if(targetDataSources != null){
            for(Object object : targetDataSources.values()){
                try {
                    if (object instanceof BasicDataSource) {
                        ((BasicDataSource) object).close();
                    } else if (object instanceof AutoCloseable) {
                        ((AutoCloseable) object).close();
                    }
                }catch(Exception e){
                    // ignore
                }
            }
        }
    }

    public BasicDataSource getOriginalDataSource() {
        return originalDataSource;
    }

    public BasicDataSource getShadeDataSource() {
        return shadeDataSource;
    }

    private Properties fetchAllProperties() {
        final Properties properties = new Properties();
        for (Iterator it = ((AbstractEnvironment) environment).getPropertySources().iterator(); it.hasNext(); ) {
            PropertySource propertySource = (PropertySource) it.next();
            if (propertySource instanceof EnumerablePropertySource) {
                for (String key : ((EnumerablePropertySource) propertySource).getPropertyNames()) {
                    properties.put(key, propertySource.getProperty(key));
                }
            }
            if (propertySource instanceof PropertiesPropertySource) {
                properties.putAll(((MapPropertySource) propertySource).getSource());
            }
            if (propertySource instanceof CompositePropertySource) {
                properties.putAll(getPropertiesInCompositePropertySource((CompositePropertySource) propertySource));
            }
        }
        return properties;
    }

    private Properties getPropertiesInCompositePropertySource(CompositePropertySource compositePropertySource) {
        final Properties properties = new Properties();
        compositePropertySource.getPropertySources().forEach(propertySource -> {
            if (propertySource instanceof EnumerablePropertySource) {
                for (String key : ((EnumerablePropertySource) propertySource).getPropertyNames()) {
                    properties.put(key, propertySource.getProperty(key));
                }
            }
            if (propertySource instanceof MapPropertySource) {
                properties.putAll(((MapPropertySource) propertySource).getSource());
            }
            if (propertySource instanceof CompositePropertySource)
                properties.putAll(getPropertiesInCompositePropertySource((CompositePropertySource) propertySource));
        });
        return properties;
    }
}
