package cn.com.duibaboot.ext.autoconfigure.web;

import cn.com.duibaboot.ext.autoconfigure.core.SpecifiedBeanPostProcessor;
import cn.com.duibaboot.ext.autoconfigure.web.cookie.Rfc2109CookieFixFilter;
import cn.com.duibaboot.ext.autoconfigure.web.mvc.*;
import org.apache.catalina.connector.ClientAbortException;
import org.apache.catalina.connector.Request;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.BeanFactory;
import org.springframework.boot.autoconfigure.AutoConfigureBefore;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnWebApplication;
import org.springframework.boot.autoconfigure.web.HttpMessageConverters;
import org.springframework.boot.autoconfigure.web.WebMvcAutoConfiguration;
import org.springframework.boot.web.servlet.FilterRegistrationBean;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.core.Ordered;
import org.springframework.http.converter.HttpMessageConverter;
import org.springframework.web.accept.ContentNegotiationManager;
import org.springframework.web.servlet.DispatcherServlet;
import org.springframework.web.servlet.config.annotation.WebMvcConfigurerAdapter;
import org.springframework.web.servlet.view.ContentNegotiatingViewResolver;

import javax.servlet.DispatcherType;
import javax.servlet.Servlet;
import java.util.ArrayList;
import java.util.Collections;
import java.util.EnumSet;
import java.util.List;

/**
 * 把ContentNegotiatingViewResolver配置为最低顺序，以防mvcAutoConfiguration自动注入最高优先级的ContentNegotiatingViewResolver
 * <br/>
 * 这样做是为了防止线上出现大量html.vm/jpg.vm/xml.vm找不到的问题。
 * 此问题重现的一种方法是在访问时先加入.html后缀，然后用正常链接访问。
 * <br/>
 * ContentNegotiatingViewResolver的大概目的是自动判断当前访问的页面，然后自动附加html等后缀寻找对应文件，我们不需要该特性
 * Created by wenqi.huang on 2017/1/5.
 */
@Configuration
@AutoConfigureBefore({WebMvcAutoConfiguration.class})
@ConditionalOnClass({ Servlet.class, DispatcherServlet.class,
        WebMvcConfigurerAdapter.class })
@ConditionalOnWebApplication
public class WebMvcFixAutoConfiguration {

    /**
     * 把ContentNegotiatingViewResolver配置为最低顺序，以防mvcAutoConfiguration自动注入最高优先级的ContentNegotiatingViewResolver
     * <br/>
     * 这样做是为了防止线上出现大量html.vm/jpg.vm/xml.vm找不到的问题。
     * 此问题重现的一种方法是在访问时先加入.html后缀，然后用正常链接访问。
     * <br/>
     * ContentNegotiatingViewResolver的大概目的是自动判断当前访问的页面，然后自动附加html等后缀寻找对应文件，我们不需要该特性
     */
    @Bean(name = "viewResolver")
    @ConditionalOnMissingBean(name = "viewResolver", value = ContentNegotiatingViewResolver.class)
    public ContentNegotiatingViewResolver viewResolver(BeanFactory beanFactory) {
        ContentNegotiatingViewResolver resolver = new ContentNegotiatingViewResolver();
        resolver.setContentNegotiationManager(
                beanFactory.getBean(ContentNegotiationManager.class));
        // ContentNegotiatingViewResolver uses all the other view resolvers to locate
        // a view so it should have a high precedence
        resolver.setOrder(Ordered.LOWEST_PRECEDENCE);
        return resolver;
    }

    /**
     * 自动加入Http Filter，对超时请求打印warn日志
     */
    @Configuration
    @ConditionalOnClass({ Servlet.class, DispatcherServlet.class,
            WebMvcConfigurerAdapter.class })
    @ConditionalOnWebApplication
    public static class dbTimeProfileFilterConfiguration{
        @Bean
        public FilterRegistrationBean dbTimeProfileFilterConfigurer(){
            DBTimeProfileFilter filter=new DBTimeProfileFilter();
            FilterRegistrationBean registrationBean = new FilterRegistrationBean();
            registrationBean.setFilter(filter);
            List<String> urlPatterns=new ArrayList<String>();
            urlPatterns.add("/*");//拦截路径，可以添加多个
            registrationBean.setUrlPatterns(urlPatterns);
            registrationBean.setDispatcherTypes(EnumSet.of(DispatcherType.REQUEST));
            registrationBean.setOrder(-1);
            return registrationBean;
        }
    }

    /**
     * 自动加入Http Filter，拦截这个异常：org.apache.catalina.connector.ClientAbortException: java.io.IOException: 断开的管道
     * <br/>
     * 改为打印warn日志
     * 避免其上报给loginsight和cat监控。
     */
    @Configuration
    @ConditionalOnClass({ Servlet.class, DispatcherServlet.class,
            WebMvcConfigurerAdapter.class, ClientAbortException.class })
    @ConditionalOnWebApplication
    public static class ClientAbortExceptionIgnoreFilterConfiguration{
        @Bean
        public FilterRegistrationBean clientAbortExceptionIgnoreFilterConfigurer(){
            ClientAbortExceptionIgnoreFilter filter = new ClientAbortExceptionIgnoreFilter();
            FilterRegistrationBean registrationBean = new FilterRegistrationBean();
            registrationBean.setFilter(filter);
            List<String> urlPatterns=new ArrayList<String>();
            urlPatterns.add("/*");//拦截路径，可以添加多个
            registrationBean.setUrlPatterns(urlPatterns);
            registrationBean.setDispatcherTypes(EnumSet.of(DispatcherType.REQUEST));
            registrationBean.setOrder(2);//在cat后面，防止不必要的异常让cat上报
            return registrationBean;
        }
    }

    /**
     * 自动加入Http Filter，修复tomcat 8.5.* 版本对RFC2109的cookie识别出来两边会带上引号的问题。
     * 修复tomcat 8.5.*版本对RFC2109的cookie处理的问题。RFC2109的cookie比如：
     * $Version="1"; _ac="eyJhaWQiOjE5NTI0LCJjaWQiOjUzMzUyMjk2OX0=";$Path="/";$Domain="duiba.com.cn"
     * <br/>
     * tomcat8.5.*版本在处理的时候会把_ac中的左右两个双引号也作为value的一部分，导致程序获取失败；tomcat8.0.*版本没有此问题。
     * 附上rfc2109的文档：https://tools.ietf.org/html/rfc2109#section-4.3.4
     */
    @Configuration
    @ConditionalOnClass({ Servlet.class, DispatcherServlet.class,
            WebMvcConfigurerAdapter.class, Request.class})
    @ConditionalOnWebApplication
    public static class Rfc2019CookieFilterConfiguration{
        @Bean
        public FilterRegistrationBean rfc2019CookieFilterConfigurer(){
            Rfc2109CookieFixFilter filter = new Rfc2109CookieFixFilter();
            FilterRegistrationBean registrationBean = new FilterRegistrationBean();
            registrationBean.setFilter(filter);
            List<String> urlPatterns=new ArrayList<>();
            urlPatterns.add("/*");//拦截路径，可以添加多个
            registrationBean.setUrlPatterns(urlPatterns);
            registrationBean.setDispatcherTypes(EnumSet.of(DispatcherType.REQUEST));
            registrationBean.setOrder(Ordered.HIGHEST_PRECEDENCE + 10);//在CharacterFilter后面
            return registrationBean;
        }
    }

    /**
     * 这个HandlerExceptionResolver用于对BizException类进行特殊处理，返回的错误json字符串中多加入一个code字段
     * @return
     */
    @Bean
    public BizExceptionResolver bizExceptionResolver(){
        return new BizExceptionResolver();
    }

    @Bean
    public CustomWebMvcRegistrations customWebMvcRegistrations(){
        return new CustomWebMvcRegistrations();
    }

    /**
     * 把HttpMessageConvertersAutoConfiguration中定义的HttpMessageConverters替换为使用自定义的HttpMessageConverters，以在最后增加fst和kryo的HttpMessageConverter.(顺序很重要，一定要在最后)
     * @return
     */
    @Bean
    public SpecifiedBeanPostProcessor httpMessageConvertersPostProcessor(){
        return new SpecifiedBeanPostProcessor<HttpMessageConverters>(){

            @Override
            public int getOrder() {
                return 0;
            }

            @Override
            public Class<HttpMessageConverters> getBeanType() {
                return HttpMessageConverters.class;
            }

            @Override
            public Object postProcessBeforeInitialization(HttpMessageConverters bean, String beanName) throws BeansException {
                return bean;
            }

            @Override
            public Object postProcessAfterInitialization(HttpMessageConverters bean, String beanName) throws BeansException {
                List<HttpMessageConverter<?>> converters = bean.getConverters();
                return new CustomHttpMessageConverters(false, converters == null
                        ? Collections.emptyList() : converters);
            }
        };
    }
}
