package cn.com.duibaboot.ext.autoconfigure.cat;

import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

import cn.com.duiba.boot.perftest.PerfTestContext;
import cn.com.duiba.boot.utils.MainApplicationContextHolder;

import org.apache.dubbo.common.URL;
import org.apache.dubbo.common.constants.CommonConstants;
import org.apache.dubbo.common.extension.Activate;
import org.apache.dubbo.rpc.Filter;
import org.apache.dubbo.rpc.Invocation;
import org.apache.dubbo.rpc.Invoker;
import org.apache.dubbo.rpc.Result;
import org.apache.dubbo.rpc.RpcContext;
import org.apache.dubbo.rpc.RpcException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import org.springframework.cloud.sleuth.ErrorParser;
import org.springframework.cloud.sleuth.Span;
import org.springframework.cloud.sleuth.SpanReporter;
import org.springframework.cloud.sleuth.SpanTextMap;
import org.springframework.cloud.sleuth.Tracer;
import org.springframework.cloud.sleuth.instrument.web.HttpSpanExtractor;
import org.springframework.cloud.sleuth.instrument.web.HttpSpanInjector;
import org.springframework.util.StringUtils;

/**
 * sleuth对dubbo的支持
 *
 * @author wenqi.huang
 */
@Activate(group = {CommonConstants.PROVIDER, CommonConstants.CONSUMER},order = 1)
public class SleuthDubboFilter implements Filter {

    private static final Logger log = LoggerFactory.getLogger(SleuthDubboFilter.class);
    private static final boolean HAS_TRACER_CLASS;

    private static volatile Tracer tracer;
    private static volatile ErrorParser errorParser;
    private static volatile SpanReporter spanReporter;

    private static volatile HttpSpanInjector httpSpanInjector;
    private static volatile HttpSpanExtractor httpSpanExtractor;

    static {
        boolean hasTracerClass = true;
        try {
            Class.forName("org.springframework.cloud.sleuth.Tracer");
        }
        catch (ClassNotFoundException e) {
            hasTracerClass = false;
        }
        HAS_TRACER_CLASS = hasTracerClass;
    }

    private static Tracer getTracer(){
        if (tracer == null) {
            if(MainApplicationContextHolder.getApplicationContext() != null) {
                tracer = MainApplicationContextHolder.getApplicationContext().getBean(Tracer.class);
            }
        }
        return tracer;
    }

    private static ErrorParser getErrorParser(){
        if (errorParser == null) {
            if(MainApplicationContextHolder.getApplicationContext() != null) {
                errorParser = MainApplicationContextHolder.getApplicationContext().getBean(ErrorParser.class);
            }
        }
        return errorParser;
    }

    private static SpanReporter getSpanReporter(){
        if (spanReporter == null) {
            if(MainApplicationContextHolder.getApplicationContext() != null) {
                spanReporter = MainApplicationContextHolder.getApplicationContext().getBean(SpanReporter.class);
            }
        }
        return spanReporter;
    }

    private static HttpSpanInjector getHttpSpanInjector(){
        if (httpSpanInjector == null) {
            if(MainApplicationContextHolder.getApplicationContext() != null) {
                httpSpanInjector = MainApplicationContextHolder.getApplicationContext().getBean(HttpSpanInjector.class);
            }
        }
        return httpSpanInjector;
    }

    private static HttpSpanExtractor getHttpSpanExtractor(){
        if (httpSpanExtractor == null) {
            if(MainApplicationContextHolder.getApplicationContext() != null) {
                httpSpanExtractor = MainApplicationContextHolder.getApplicationContext().getBean(HttpSpanExtractor.class);
            }
        }
        return httpSpanExtractor;
    }

    @Override
    public Result invoke(Invoker<?> invoker, Invocation invocation) throws RpcException {
        if (!HAS_TRACER_CLASS) {
            return invoker.invoke(invocation);
        }
        if (getTracer() == null) {
            return invoker.invoke(invocation);
        }

        URL url = invoker.getUrl();
        String sideKey = url.getParameter(CommonConstants.SIDE_KEY);
        String spanName = "dubbo:/" + invoker.getInterface().getSimpleName()+"/"+invocation.getMethodName();
        if (CommonConstants.CONSUMER_SIDE.equals(sideKey)) {
            Span span = getTracer().createSpan(spanName);
            try {
                getHttpSpanInjector().inject(span, new DubboClientRequestTextMap(RpcContext.getContext()));
                if(span.isExportable()) {
                    span.tag(Span.SPAN_LOCAL_COMPONENT_TAG_NAME, "dubbo");//本地组件名
//                span.tag(Span.SPAN_PEER_SERVICE_TAG_NAME, "dubbo");//远程服务名
                    span.tag("peer.host", RpcContext.getContext().getRemoteHost() + ":" + RpcContext.getContext().getRemotePort());//远程host
                    span.tag("dubbo.invokeMethod", invoker.getInterface().getSimpleName() + "." + invocation.getMethodName());//method
                    span.tag("client.isPerfTest", Boolean.toString(PerfTestContext.isCurrentInPerfTestMode()));
                    span.tag("client.thread", Thread.currentThread().getName());
                }
                span.logEvent(Span.CLIENT_SEND);

                Result result = invoker.invoke(invocation);
                if(result.hasException()){
                    getErrorParser().parseErrorTags(span, result.getException());
                }
                return result;
            } catch(Exception e){
                getErrorParser().parseErrorTags(span, e);
                throw e;
            } finally {
                span.logEvent(Span.CLIENT_RECV);
                getTracer().close(span);
            }
        } else { //PROVIDER_SIDE
            Span span = getHttpSpanExtractor().joinTrace(new DubboServerRequestTextMap(RpcContext.getContext()));
            if (span != null) {
                getTracer().continueSpan(span);
            } else {
                span = getTracer().createSpan(spanName);
            }

            try {
                if(span.isExportable()) {
                    span.tag(Span.SPAN_LOCAL_COMPONENT_TAG_NAME, "dubbo");//本地组件名
                    span.tag("server.isPerfTest", Boolean.toString(PerfTestContext.isCurrentInPerfTestMode()));
                    span.tag("server.thread", Thread.currentThread().getName());
//                span.tag(Span.SPAN_PEER_SERVICE_TAG_NAME, "dubbo");//远程服务名
                }
                span.logEvent(Span.SERVER_RECV);

                Result result = invoker.invoke(invocation);
                if(result.hasException()){
                    getErrorParser().parseErrorTags(span, result.getException());
                }
                return result;
            } catch(Exception e){
                getErrorParser().parseErrorTags(span, e);
                throw e;
            } finally {
                span.logEvent(Span.SERVER_SEND);
                recordParentSpan(span);
                getTracer().close(span);
            }
        }
    }

    private void recordParentSpan(Span parent) {
        if (parent == null) {
            return;
        }
        if (parent.isRemote()) {
            if (log.isDebugEnabled()) {
                log.debug("Trying to send the parent span " + parent + " to Zipkin");
            }
            parent.stop();
            // should be already done by HttpServletResponse wrappers
            getSpanReporter().report(parent);
        }
    }

    static class DubboClientRequestTextMap implements SpanTextMap {

        private final RpcContext delegate;

        DubboClientRequestTextMap(RpcContext delegate) {
            this.delegate = delegate;
        }

        @Override
        public Iterator<Map.Entry<String, String>> iterator() {
            Map<String, String> map = new HashMap<>();
            //返回空map，以允许覆盖trace相关的头。
            return map.entrySet().iterator();
        }

        @Override
        public void put(String key, String value) {
            if (!StringUtils.hasText(value)) {
                return;
            }
            delegate.setAttachment(key, value);
        }
    }

    static class DubboServerRequestTextMap implements SpanTextMap {

        private final RpcContext delegate;

        DubboServerRequestTextMap(RpcContext delegate) {
            this.delegate = delegate;
        }

        @Override
        public Iterator<Map.Entry<String, String>> iterator() {
            Map<String, String> map = new HashMap<>();
            map.putAll(delegate.getAttachments());
            map.put("X-Span-Uri", "/undefined");//ZipkinHttpSpanMapper.URI_HEADER="X-Span-Uri",添加这个字段是为了防止ZipkinHttpSpanExtractor空指针报错。
            return map.entrySet().iterator();
        }

        @Override
        public void put(String key, String value) {
            if (!StringUtils.hasText(value)) {
                return;
            }
            delegate.setAttachment(key, value);
        }
    }
}
