package cn.com.duibaboot.ext.autoconfigure.cat;

import cn.com.duiba.boot.perftest.PerfTestContext;
import org.aopalliance.intercept.MethodInterceptor;
import org.aopalliance.intercept.MethodInvocation;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.reflect.MethodSignature;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.aop.framework.ProxyFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cloud.sleuth.ErrorParser;
import org.springframework.cloud.sleuth.Span;
import org.springframework.cloud.sleuth.Tracer;
import org.springframework.core.annotation.Order;
import org.springframework.data.redis.connection.jedis.JedisConnection;

/**
 * 加入aop，监控spring-data-redis执行耗时
 */
@Aspect
@Order(-1)
public class SleuthSpringDataRedisPlugin {

	private static final Logger logger = LoggerFactory.getLogger(SleuthSpringDataRedisPlugin.class);

	@Autowired
	private Tracer tracer;
	@Autowired
	private ErrorParser errorParser;

	private volatile Class<?> lastJedisConnectionProxyClass;

	@Around("execution(* org.springframework.data.redis.connection.jedis.JedisConnectionFactory.getConnection(..))")
	public Object springDataRedisJoinPoint(ProceedingJoinPoint joinPoint) throws Throwable{
		if(!tracer.isTracing() || !tracer.getCurrentSpan().isExportable()){
			return joinPoint.proceed();
		}

		MethodSignature signature = (MethodSignature) joinPoint.getSignature();
		String methodName = signature.getMethod().getName();

		if("getConnection".equals(methodName)){
			try {
				JedisConnection connection = (JedisConnection) joinPoint.proceed();

				ProxyFactory factory = new ProxyFactory();
				factory.setTarget(connection);
				//factory.setTargetClass(JedisConnection.class);
				factory.addAdvice(new JedisConnectionMethodInterceptor());
				JedisConnection newConnection = (JedisConnection) factory.getProxy();//getClass().getClassLoader()

				if(lastJedisConnectionProxyClass != null && lastJedisConnectionProxyClass != newConnection.getClass()){
					logger.error("JedisConnectionProxyClass is not same，this is spring's bug,please upgrade spring-boot's version to 1.3.8.RELEASE or higher! {},{},{},{}",new Object[]{getClass().getClassLoader(), newConnection.getClass().getClassLoader(),lastJedisConnectionProxyClass, newConnection.getClass()});//如果抱这个错，则有perm区内存溢出的风险，需要关注
				}
				lastJedisConnectionProxyClass = newConnection.getClass();

				return newConnection;
			} catch (Throwable e) {
				//失败才记录
				Span span = tracer.createSpan("redis:/"+methodName);
				if(span.isExportable()) {
					span.tag("redis.op", methodName);// get/...
					span.tag(Span.SPAN_LOCAL_COMPONENT_TAG_NAME, "springDataRedis");//本地组件名
					span.tag(Span.SPAN_PEER_SERVICE_TAG_NAME, "redis");//远程服务名
					span.tag("isPerfTest", Boolean.toString(PerfTestContext.isCurrentInPerfTestMode()));
					span.tag("thread", Thread.currentThread().getName());
//            span.tag("peer.host", "www.duiba.com.cn:3306");//远程host
					errorParser.parseErrorTags(span, e);
				}

				this.tracer.close(span);

				throw e;
			}
		}else{
			return joinPoint.proceed();
		}
	}

	/**
	 * {@link MethodInterceptor}
	 */
	private class JedisConnectionMethodInterceptor implements MethodInterceptor {

		JedisConnectionMethodInterceptor() {
		}

		@Override
		public Object invoke(MethodInvocation invocation) throws Throwable {
			String methodName = invocation.getMethod().getName();

			if(methodName.equals("isPipelined") || methodName.equals("openPipeline") || methodName.equals("isQueueing")
					|| methodName.equals("isClosed")
					|| methodName.equals("close")
					|| methodName.equals("closePipeline")){
				return invocation.proceed();
			}else if(methodName.equals("getNativeConnection")){
				Object nativeConnection = invocation.proceed();

				ProxyFactory factory = new ProxyFactory();
				factory.setTarget(nativeConnection);
				//factory.setTargetClass(JedisConnection.class);
				factory.addAdvice(new JedisMethodInterceptor());
				Object newNativeConnection = factory.getProxy();//getClass().getClassLoader()

				return newNativeConnection;
			}

			return interceptorRedisMethods(invocation);
		}

	}

	/**
	 * {@link MethodInterceptor}
	 */
	private class JedisMethodInterceptor implements MethodInterceptor {

		JedisMethodInterceptor() {
		}

		@Override
		public Object invoke(MethodInvocation invocation) throws Throwable {
			return interceptorRedisMethods(invocation);
		}

	}

	private Object interceptorRedisMethods(MethodInvocation invocation) throws Throwable {
		String methodName = invocation.getMethod().getName();
		//忽略Object基类中的方法
		if(methodName.equals("toString") || methodName.equals("hashCode") || methodName.equals("equals")){
			return invocation.proceed();
		}


		Span span = tracer.createSpan("redis:/"+methodName);
		try {
			if(span.isExportable()) {
				span.tag("redis.op", methodName);// get/...
				span.tag(Span.SPAN_LOCAL_COMPONENT_TAG_NAME, "springDataRedis");//本地组件名
				span.tag(Span.SPAN_PEER_SERVICE_TAG_NAME, "redis");//远程服务名
//            span.tag("peer.host", "www.duiba.com.cn:3306");//远程host
				if ("get".equals(methodName)
						&& invocation.getArguments() != null
						&& invocation.getArguments().length == 1
						&& invocation.getArguments()[0] != null) {
					Object obj = invocation.getArguments()[0];
					if (obj instanceof byte[]) {
						span.tag("redis.key", new String((byte[]) obj));//key
					}
					else {
						span.tag("redis.key", obj.toString());//key
					}
				}
				span.tag("isPerfTest", Boolean.toString(PerfTestContext.isCurrentInPerfTestMode()));
				span.tag("thread", Thread.currentThread().getName());
			}
			span.logEvent(Span.CLIENT_SEND);

			return invocation.proceed();
		} catch(Exception e){
			errorParser.parseErrorTags(span, e);
			throw e;
		} finally {
			span.logEvent(Span.CLIENT_RECV);
			this.tracer.close(span);
		}
	}

}
