package cn.com.duibaboot.ext.autoconfigure.flowreplay;

import cn.com.duibaboot.ext.autoconfigure.core.SpecifiedBeanPostProcessor;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.record.RecordResultBuilder;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.record.RecordTraceWriter;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.record.aop.*;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.record.endpoint.RecordMvcEndpoint;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.record.endpoint.RecordResultMvcEndpoint;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.record.endpoint.RecordStopMvcEndpoint;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.record.sampler.PercentageBasedRecordSampler;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.record.sampler.RecordSampler;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.replay.*;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.replay.aop.*;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.replay.endpoint.ReplayMvcEndpoint;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.replay.endpoint.ReplayResultMvcEndpoint;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.replay.endpoint.ReplayStopMvcEndpoint;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.replay.endpoint.ReplaySysTimeMvcEndpoint;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.replay.replayer.ConcreateReplayer;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.replay.replayer.RemoteServiceReplayer;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.replay.replayer.Replayer;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.replay.replayer.SpringMvcReplayer;
import cn.com.duibaboot.ext.autoconfigure.security.BaseSecurityManager;
import cn.com.duibaboot.ext.autoconfigure.security.SecurityManagerAutoConfiguration;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.http.client.HttpClient;
import org.apache.ibatis.plugin.Interceptor;
import org.apache.ibatis.session.SqlSessionFactory;
import org.apache.rocketmq.client.producer.DefaultMQProducer;
import org.mybatis.spring.SqlSessionFactoryBean;
import org.mybatis.spring.SqlSessionTemplate;
import org.springframework.beans.BeansException;
import org.springframework.boot.autoconfigure.condition.ConditionalOnBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnWebApplication;
import org.springframework.boot.web.servlet.FilterRegistrationBean;
import org.springframework.cloud.client.loadbalancer.RestTemplateCustomizer;
import org.springframework.context.ApplicationListener;
import org.springframework.context.annotation.*;
import org.springframework.context.event.ContextRefreshedEvent;
import org.springframework.core.Ordered;
import org.springframework.core.annotation.Order;
import org.springframework.core.type.AnnotatedTypeMetadata;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.http.client.HttpComponentsClientHttpRequestFactory;
import org.springframework.http.converter.ByteArrayHttpMessageConverter;
import org.springframework.web.client.RestTemplate;
import org.springframework.web.servlet.config.annotation.InterceptorRegistry;
import org.springframework.web.servlet.config.annotation.WebMvcConfigurerAdapter;

import javax.annotation.Resource;
import javax.servlet.DispatcherType;
import javax.servlet.Servlet;
import java.security.Permission;
import java.util.EnumSet;
import java.util.List;

/**
 * 引流回归自动配置
 * Created by guoyanfei .
 * 2019-01-23 .
 */
@Lazy
@Configuration
public class FlowReplayAutoConfiguration {

    /**
     * 录制配置
     */
    @Configuration
    @Conditional({ AgentNormalCondition.class })
    static class RecordConfiguration {

        @Configuration
        @ConditionalOnClass({DefaultMQProducer.class})
        protected static class RecordRocketMqProducerPluginConfiguration {

            @Bean
            public RecordRocketMqProducerPlugin recordRocketMqProducerPlugin() {
                return new RecordRocketMqProducerPlugin();
            }

        }

        @Configuration
        @ConditionalOnBean({RedisTemplate.class})
        protected static class RecordRedisTemplatePluginConfiguration {

            @Bean
            public RecordRedisTemplatePlugin recordRedisTemplatePlugin() {
                return new RecordRedisTemplatePlugin();
            }
        }

        @Bean
        public RecordCustomizeFlowReplaySpanPlugin recordCustomizeFlowReplaySpanPlugin() {
            return new RecordCustomizeFlowReplaySpanPlugin();
        }

        /**
         * /flow/record 端点
         * 开始录制
         *
         * @return
         */
        @Bean
        public RecordMvcEndpoint recordMvcEndpoint() {
            return new RecordMvcEndpoint();
        }

        /**
         * /flow/record/stop 端点
         * 结束录制
         *
         * @return
         */
        @Bean
        public RecordStopMvcEndpoint recordStopMvcEndpoint() {
            return new RecordStopMvcEndpoint();
        }

        /**
         * /flow/record/result 端点
         * 查询录制结果
         *
         * @return
         */
        @Bean
        public RecordResultMvcEndpoint recordResultMvcEndpoint() {
            return new RecordResultMvcEndpoint();
        }

        /**
         * 录制结果构建器
         * 录制的用例集文件压缩上传，并且构建结果回填给 RecordContext
         *
         * @return
         */
        @Bean
        public RecordResultBuilder recordResultBuilder() {
            return new RecordResultBuilder();
        }

        /**
         * SpringMvc 控制器方法的过滤器
         * 录制http的请求
         *
         * @return
         */
        @Bean
        public RecordSpringMvcFilter recordSpringMvcFilter() {
            return new RecordSpringMvcFilter();
        }

        /**
         * 录制结果写到文件中
         * 把内存队列中的录制结果写到文件中
         *
         * @return
         */
        @Bean
        public RecordTraceWriter recordTraceWriter() {
            return new RecordTraceWriter();
        }

        @Bean
        public FilterRegistrationBean recordSpringMvcFilterConfigurer(RecordSpringMvcFilter recordSpringMvcFilter) {
            FilterRegistrationBean registrationBean = new FilterRegistrationBean();
            registrationBean.setFilter(recordSpringMvcFilter);
            List<String> urlPatterns = Lists.newArrayList();
            urlPatterns.add("/*");//拦截路径，可以添加多个
            registrationBean.setUrlPatterns(urlPatterns);
            registrationBean.setDispatcherTypes(EnumSet.of(DispatcherType.REQUEST));
            registrationBean.setOrder(0);
            return registrationBean;
        }

        @Configuration
        @ConditionalOnClass({Servlet.class, RecordExcludeHandlerInterceptor.class, InterceptorRegistry.class})
        @ConditionalOnWebApplication
        @Order(-100)
        public static class RecordExcludeHandlerInterceptorConfiguration extends WebMvcConfigurerAdapter {

            @Override
            public void addInterceptors(InterceptorRegistry registry) {
                RecordExcludeHandlerInterceptor interceptor = new RecordExcludeHandlerInterceptor();
                registry.addInterceptor(interceptor);
                super.addInterceptors(registry);
            }

        }

        @Configuration
        static class FlowRecordAspectConfiguration {

            /**
             * remoteService录制拦截器配置
             *
             * @return
             */
            @Bean
            public RecordRemoteServicePlugin recordRemoteServicePlugin() {
                return new RecordRemoteServicePlugin();
            }

        }

        /**
         * 录制采样器配置
         */
        @Bean
        @ConditionalOnMissingBean
        public RecordSampler recordSampler() {
            return new PercentageBasedRecordSampler();
        }

        @Configuration
        @ConditionalOnClass({ SqlSessionTemplate.class, SqlSessionFactoryBean.class, SqlSessionFactory.class, RecordMybatisPlugin.class })
        public static class RecordMyBatisPostProcessorConfiguration {

            @Bean
            public RecordMybatisPlugin recordMybatisPlugin() {
                return new RecordMybatisPlugin();
            }

            /**
             * 声明后置处理器，spring全部bean初始化完成后调用，给所有SqlSessionBean注入RecordMybatisPlugin plugin
             *
             * @return
             */
            @Bean
            public static SpecifiedBeanPostProcessor recordMyBatisPostProcessorConfigurer() {
                return new RecordMybatisBeanPostProcessor();
            }
        }

        private static class RecordMybatisBeanPostProcessor implements SpecifiedBeanPostProcessor<Object> {

            @Resource
            private RecordMybatisPlugin recordMybatisPlugin;

            @Override
            public int getOrder() {
                return -10;
            }

            @Override
            public Class<Object> getBeanType() {
                return Object.class;
            }

            @Override
            public Object postProcessBeforeInitialization(Object bean, String beanName) throws BeansException {
                return bean;
            }

            @Override
            public Object postProcessAfterInitialization(Object bean, String beanName) throws BeansException {
                SqlSessionFactory s = null;
                if (bean instanceof SqlSessionFactory) {
                    s = (SqlSessionFactory) bean;
                }
                if (bean instanceof SqlSessionTemplate) {
                    s = ((SqlSessionTemplate) bean).getSqlSessionFactory();
                }
                if (s == null) {
                    return bean;
                }

                addInterceptor(s);

                return bean;
            }

            private void addInterceptor(SqlSessionFactory s) {
                boolean hasPlugin = false;
                if (!CollectionUtils.isEmpty(s.getConfiguration().getInterceptors())) {
                    for (Interceptor plugin : s.getConfiguration().getInterceptors()) {
                        if (plugin instanceof RecordMybatisPlugin) {
                            hasPlugin = true;
                            break;
                        }
                    }
                }

                if (!hasPlugin) {
                    s.getConfiguration().addInterceptor(recordMybatisPlugin);
                }
            }
        }
    }

    /**
     * 回归配置
     */
    @Configuration
    @Conditional({ ReplayCondition.class, AgentNormalCondition.class })
    static class ReplayConfiguration {

        @Configuration
        @ConditionalOnClass({DefaultMQProducer.class})
        protected static class ReplayRocketMqProducerPluginConfiguration {

            @Bean
            public ReplayRocketMqProducerPlugin replayRocketMqProducerPlugin() {
                return new ReplayRocketMqProducerPlugin();
            }
        }

        @Configuration
        @ConditionalOnBean({RedisTemplate.class})
        protected static class ReplayRedisTemplatePluginConfiguration {

            @Bean
            public ReplayRedisTemplatePlugin replayRedisTemplatePlugin() {
                return new ReplayRedisTemplatePlugin();
            }
        }

        @Bean
        public ReplayCustomizeFlowReplaySpanPlugin replayCustomizeFlowReplaySpanPlugin() {
            return new ReplayCustomizeFlowReplaySpanPlugin();
        }

        /**
         * /flow/repaly 端点
         * 开始回归
         *
         * @return
         */
        @Bean
        public ReplayMvcEndpoint replayMvcEndpoint() {
            return new ReplayMvcEndpoint();
        }

        /**
         * /flow/replay/stop 端点
         * 终止回归
         *
         * @return
         */
        @Bean
        public ReplayStopMvcEndpoint replayStopMvcEndpoint() {
            return new ReplayStopMvcEndpoint();
        }

        /**
         * /flow/replay/systime
         * @return
         */
        @Bean
        public ReplaySysTimeMvcEndpoint replaySysTimeMvcEndpoint() {
            return new ReplaySysTimeMvcEndpoint();
        }

        /**
         * /flow/replay/result 端点
         * 查询回归结果
         *
         * @return
         */
        @Bean
        public ReplayResultMvcEndpoint replayResultMvcEndpoint() {
            return new ReplayResultMvcEndpoint();
        }

        /**
         * 用例集加载器的配置
         * 下载并解压录制的用例集文件，然后把用例读取到内存中，供用例消费者消费
         *
         * @return
         */
        @Bean
        public ReplayTraceLoader replayTraceLoader() {
            return new ReplayTraceLoader();
        }

        /**
         * 用例消费者配置
         * 多线程消费用例加载器加载到内存中的用例，调用回归器回归对应类型的用例
         *
         * @return
         */
        @Bean
        public ReplayTraceReplayer replayTraceReplayer() {
            return new ReplayTraceReplayer();
        }

        /**
         * 回归结果写到文件中
         * 把内存队列中的回归结果写到文件中
         *
         * @return
         */
        @Bean
        public ReplayTraceResultWriter replayTraceResultWriter() {
            return new ReplayTraceResultWriter();
        }

        /**
         * 回归结果构建器
         * 回归的结果文件压缩上传，并且构建结果回填给 ReplayContext
         *
         * @return
         */
        @Bean
        public ReplayResultBuilder replayResultBuilder() {
            return new ReplayResultBuilder();
        }

        /**
         * 回归器配置
         * 目前支持 RemoteService 和 SpringMvc Controller两种调用的回归
         */
        @Configuration
        static class ReplayerConfiguration {

            @Bean
            public RemoteServiceReplayer remoteServiceReplayer() {
                return new RemoteServiceReplayer();
            }

            @Bean
            public SpringMvcReplayer springMvcReplayer() {
                return new SpringMvcReplayer();
            }

            @Bean
            public Replayer replayer() {
                return new ConcreateReplayer();
            }

        }

        /**
         * SpringMvc 控制器方法的过滤器
         * 回归http的请求设置上下文
         *
         * @return
         */
        @Bean
        public ReplaySpringMvcFilter replaySpringMvcFilter() {
            return new ReplaySpringMvcFilter();
        }

        @Bean
        public FilterRegistrationBean replaySpringMvcFilterConfigurer(ReplaySpringMvcFilter replaySpringMvcFilter) {
            FilterRegistrationBean registrationBean = new FilterRegistrationBean();
            registrationBean.setFilter(replaySpringMvcFilter);
            List<String> urlPatterns = Lists.newArrayList();
            urlPatterns.add("/*");//拦截路径，可以添加多个
            registrationBean.setUrlPatterns(urlPatterns);
            registrationBean.setDispatcherTypes(EnumSet.of(DispatcherType.REQUEST));
            registrationBean.setOrder(0);
            return registrationBean;
        }

        /**
         * 设置SecurityManager，以支持禁止访问部分网络的功能
         *
         * @return
         */
        @Bean
        public DuibaSecurityManagerConfiguar4ReplayApplicationListener duibaSecurityManagerConfiguar4ReplayApplicationListener() {
            return new DuibaSecurityManagerConfiguar4ReplayApplicationListener();
        }

        @Slf4j
        static class DuibaSecurityManagerConfiguar4ReplayApplicationListener implements ApplicationListener<ContextRefreshedEvent>, Ordered {

            private boolean flag = true;

            @Override
            public void onApplicationEvent(ContextRefreshedEvent applicationStartedEvent) {
                if (!flag) {
                    return;
                }
                try {
                    System.setSecurityManager(new BaseSecurityManager() {
                        @Override
                        public void checkExec(String cmd) {
                            //允许执行任意shell
                        }

                        @Override
                        public void checkPermission(Permission perm) {
                            //允许任意Permission, 除了不允许调用setSecurityManager
                            if ("setSecurityManager".equals(perm.getName())) {
                                super.checkPermission(perm);
                            }
                        }

                        @Override
                        public void checkAccept(String host, int port) {
                            if (ReplayContextHolder.canHostPass(host)) {
                                return;
                            }
                            throw new SecurityException("Not allowed to accept to host:" + host + ",port:" + port);
                        }

                        @Override
                        public void checkConnect(String host, int port) {//判断是否可以连接远程host
                            if (ReplayContextHolder.canHostPass(host)) {
                                return;
                            }
                            throw new SecurityException("Not allowed to connect to host:" + host + ",port:" + port);
                        }

                        @Override
                        public void checkConnect(String host, int port, Object context) {
                            this.checkConnect(host, port);

                        }
                    });
                } catch (Throwable e) {
                    log.error("设置SercurityManager失败，这会导致回归时仍然能访问网络，可能会污染数据库数据，退出当前应用", e);
                    System.exit(-1);
                }

                flag = false;
            }

            @Override
            public int getOrder() {
                return SecurityManagerAutoConfiguration.ORDER_OF_CONTEXT_REFRESHED_EVENT - 1;
            }
        }

        @Configuration
        @ConditionalOnClass({ SqlSessionTemplate.class, SqlSessionFactoryBean.class, SqlSessionFactory.class, ReplayMybatisPlugin.class })
        public static class ReplayMyBatisPostProcessorConfiguration {

            @Bean
            public ReplayMybatisPlugin replayMybatisPlugin() {
                return new ReplayMybatisPlugin();
            }

            /**
             * 声明后置处理器，spring全部bean初始化完成后调用，给所有SqlSessionBean注入RecordMybatisPlugin plugin
             *
             * @return
             */
            @Bean
            public static SpecifiedBeanPostProcessor replayMyBatisPostProcessorConfigurer() {
                return new ReplayMybatisBeanPostProcessor();
            }
        }

        private static class ReplayMybatisBeanPostProcessor implements SpecifiedBeanPostProcessor<Object> {

            @Resource
            private ReplayMybatisPlugin replayMybatisPlugin;

            @Override
            public int getOrder() {
                return -10;
            }

            @Override
            public Class<Object> getBeanType() {
                return Object.class;
            }

            @Override
            public Object postProcessBeforeInitialization(Object bean, String beanName) throws BeansException {
                return bean;
            }

            @Override
            public Object postProcessAfterInitialization(Object bean, String beanName) throws BeansException {
                SqlSessionFactory s = null;
                if (bean instanceof SqlSessionFactory) {
                    s = (SqlSessionFactory) bean;
                }
                if (bean instanceof SqlSessionTemplate) {
                    s = ((SqlSessionTemplate) bean).getSqlSessionFactory();
                }
                if (s == null) {
                    return bean;
                }

                addInterceptor(s);

                return bean;
            }

            private void addInterceptor(SqlSessionFactory s) {
                boolean hasPlugin = false;
                if (!CollectionUtils.isEmpty(s.getConfiguration().getInterceptors())) {
                    for (Interceptor plugin : s.getConfiguration().getInterceptors()) {
                        if (plugin instanceof ReplayMybatisPlugin) {
                            hasPlugin = true;
                            break;
                        }
                    }
                }

                if (!hasPlugin) {
                    s.getConfiguration().addInterceptor(replayMybatisPlugin);
                }
            }
        }
    }

    private static class ReplayCondition implements Condition {

        @Override
        public boolean matches(ConditionContext context, AnnotatedTypeMetadata metadata) {
            return FlowReplayUtils.isReplayEnv();
        }
    }

    private static class AgentNormalCondition implements Condition {

        @Override
        public boolean matches(ConditionContext context, AnnotatedTypeMetadata metadata) {
            return !"true".equals(System.getProperty("duiba.noagent"));
        }
    }

    /**
     * 文件操作组件
     * 使用 loadBalancedRestTemplate
     *
     * @param httpClient
     * @param customizers
     * @return
     */
    @Bean
    public FlowReplayFileComponent flowReplayFileComponent(HttpClient httpClient, List<RestTemplateCustomizer> customizers) {
        RestTemplate loadBalancedRestTemplate = loadBalancedRestTemplate(httpClient);
        for (RestTemplateCustomizer customizer : customizers) {
            customizer.customize(loadBalancedRestTemplate);
        }
        return new FlowReplayFileComponent(loadBalancedRestTemplate);
    }

    private RestTemplate loadBalancedRestTemplate(HttpClient httpClient) {
        HttpComponentsClientHttpRequestFactory httpRequestFactory = new HttpComponentsClientHttpRequestFactory();
        httpRequestFactory.setConnectionRequestTimeout(3000);
        httpRequestFactory.setConnectTimeout(3000);
        httpRequestFactory.setReadTimeout(60000);
        httpRequestFactory.setHttpClient(httpClient);
        RestTemplate template = new RestTemplate(httpRequestFactory);
        template.getMessageConverters().add(new ByteArrayHttpMessageConverter());
        return template;
    }

}
