package cn.com.duibaboot.ext.autoconfigure.cat;

import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Optional;

import cn.com.duiba.boot.perftest.PerfTestContext;
import cn.com.duibaboot.ext.autoconfigure.grouping.ServiceGroupContext;
import org.apache.rocketmq.common.message.Message;
import org.apache.rocketmq.common.message.MessageExt;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.reflect.MethodSignature;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cloud.sleuth.ErrorParser;
import org.springframework.cloud.sleuth.Span;
import org.springframework.cloud.sleuth.SpanReporter;
import org.springframework.cloud.sleuth.SpanTextMap;
import org.springframework.cloud.sleuth.Tracer;
import org.springframework.cloud.sleuth.instrument.messaging.MessagingSpanTextMapExtractor;
import org.springframework.cloud.sleuth.instrument.messaging.MessagingSpanTextMapInjector;
import org.springframework.util.StringUtils;

/**
 * sleuth对rocketmq的支持
 */
@Aspect
public class SleuthRocketMqPlugin {

	private static final Logger log = LoggerFactory.getLogger(SleuthRocketMqPlugin.class);

	@Autowired
	private Tracer tracer;
	@Autowired
	private ErrorParser errorParser;
	@Autowired
	private SpanReporter spanReporter;
	@Autowired
	private MessagingSpanTextMapInjector messagingSpanTextMapInjector;
	@Autowired
	private MessagingSpanTextMapExtractor messagingSpanTextMapExtractor;

	@Around("execution(* org.apache.rocketmq.client.producer.DefaultMQProducer.send(..))")
	public Object rocketMqSendJoinPoint(ProceedingJoinPoint joinPoint) throws Throwable{
		if(!tracer.isTracing()){
			return joinPoint.proceed();
		}

		MethodSignature signature = (MethodSignature) joinPoint.getSignature();
		String methodName = signature.getMethod().getName();
		Object[] args = joinPoint.getArgs();
		if(args == null || args.length < 1 || args[0] == null || !(args[0] instanceof Message)) {
			return joinPoint.proceed();
		}

		Span span = tracer.createSpan("rocketmq-send");
		try {
			Message msg = (Message) args[0];
			messagingSpanTextMapInjector.inject(span, new RocketMqRequestTextMap(msg));
			if(span.isExportable()) {
				span.tag(Span.SPAN_LOCAL_COMPONENT_TAG_NAME, "rocketmqProducer");//本地组件名
				span.tag(Span.SPAN_PEER_SERVICE_TAG_NAME, "rocketmq");//远程服务名
//            span.tag("peer.host", "www.duiba.com.cn:3306");//远程host
				span.tag("rocketmq.topic", msg.getTopic());//topic
				span.tag("rocketmq.tags", msg.getTags());//tags
				span.tag("rocketmq.type", "send");
				span.tag("producer.isPerfTest", Boolean.toString(PerfTestContext.isCurrentInPerfTestMode()));
				span.tag("producer.thread", Thread.currentThread().getName());
				span.tag("producer.serviceGroupKey", ServiceGroupContext.getCurrentGroupKey());
			}
			span.logEvent(Span.CLIENT_SEND);

			return joinPoint.proceed();
		} catch(Exception e){
			errorParser.parseErrorTags(span, e);
			throw e;
		} finally {
			span.logEvent(Span.CLIENT_RECV);
			this.tracer.close(span);
		}
	}

	@Around("execution(* org.apache.rocketmq.client.consumer.listener.MessageListener*.consumeMessage(..))")
	public Object rocketMqConsumeJoinPoint(ProceedingJoinPoint joinPoint) throws Throwable{
		MethodSignature signature = (MethodSignature) joinPoint.getSignature();
		String methodName = signature.getMethod().getName();
		Object[] args = joinPoint.getArgs();
		if(args == null || args.length < 1 || args[0] == null || !(args[0] instanceof List)) {
			return joinPoint.proceed();
		}
		List<?> list = (List<?>)args[0];
		if(list == null || list.isEmpty() || !(list.get(0) instanceof MessageExt)) {
			return joinPoint.proceed();
		}

		MessageExt msg = (MessageExt)list.get(0);
		Span span = null;
		if(list.size() == 1) {
			span = messagingSpanTextMapExtractor.joinTrace(new RocketMqRequestTextMap(msg));
			if(span != null) {
				span = tracer.createSpan("rocketmq-consume", span);
			}
		}
		if(span == null) {
			span = tracer.createSpan("rocketmq-consume");
		}

		try {
			if(span.isExportable()) {
				span.tag(Span.SPAN_LOCAL_COMPONENT_TAG_NAME, "rocketmqConsumer");//本地组件名
				span.tag(Span.SPAN_PEER_SERVICE_TAG_NAME, "rocketmq");//远程服务名
//            span.tag("peer.host", "www.duiba.com.cn:3306");//远程host
				span.tag("rocketmq.topic", msg.getTopic());//topic
				span.tag("rocketmq.tags", msg.getTags());//tags
				span.tag("rocketmq.type", "consume");
				span.tag("consumer.thread", Thread.currentThread().getName());
				//TODO 可能由于aop顺序不对的原因，consumer.isPerfTest、consumer.serviceGroupKey有可能打印出错误的值
				span.tag("consumer.isPerfTest", Boolean.toString(PerfTestContext.isCurrentInPerfTestMode()));
				span.tag("consumer.serviceGroupKey", ServiceGroupContext.getCurrentGroupKey());
			}
			span.logEvent(Span.SERVER_RECV);

			return joinPoint.proceed();
		} catch(Exception e){
			errorParser.parseErrorTags(span, e);
			throw e;
		} finally {
			span.logEvent(Span.SERVER_SEND);

			recordParentSpan(span);
			this.tracer.close(span);
		}
	}

	private void recordParentSpan(Span parent) {
		if (parent == null) {
			return;
		}
		if (parent.isRemote()) {
			if (log.isDebugEnabled()) {
				log.debug("Trying to send the parent span " + parent + " to Zipkin");
			}
			parent.stop();
			// should be already done by HttpServletResponse wrappers
			spanReporter.report(parent);
		}
	}

	class RocketMqRequestTextMap implements SpanTextMap {

		private final Message delegate;

		RocketMqRequestTextMap(Message delegate) {
			this.delegate = delegate;
		}

		@Override
		public Iterator<Map.Entry<String, String>> iterator() {
			return Optional.ofNullable(delegate.getProperties()).orElse(Collections.emptyMap())
					.entrySet().iterator();
		}

		@Override
		public void put(String key, String value) {
			if (!StringUtils.hasText(value)) {
				return;
			}
			addHeader(key, value);
		}

		private void addHeader(String key, String value) {
			delegate.putUserProperty(key, value);
		}
	}

}
