package cn.com.duibaboot.ext.autoconfigure.flowreplay.span;

import cn.com.duiba.wolf.utils.UUIDUtils;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.FlowReplayException;
import org.apache.commons.lang3.StringUtils;
import org.apache.http.*;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpUriRequest;
import org.apache.http.message.BasicHttpRequest;
import org.apache.http.util.EntityUtils;

import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.net.URLDecoder;
import java.util.*;

/**
 * Created by guoyanfei .
 * 2019-07-18 .
 */
public class HttpClientFlowReplaySpan extends FlowReplaySpan {

    private static final long serialVersionUID = -7977781583338574639L;

    private String method;

    private String url;

    private Map<String, List<String>> requestHeaders;

    private Map<String, List<String>> requestParameters;

    private byte[] requestBody;

    private Map<String, List<String>> responseHeaders;

    /** Name of the protocol. */
    private String protocol;

    /** Major version number of the protocol */
    private int major;

    /** Minor version number of the protocol */
    private int minor;

    private int statusCode;

    private String reasonPhrase;

    private byte[] responseBody;

    @Override
    public SpanType getSpanType() {
        return SpanType.HTTP_CLIENT;
    }

    /**
     * 创建span
     * @param httpRequest
     * @param response
     * @return
     * @throws IOException
     */
    public static HttpClientFlowReplaySpan createSpan(HttpHost httpHost, HttpRequest httpRequest, CloseableHttpResponse response) throws IOException {
        HttpClientFlowReplaySpan span = new HttpClientFlowReplaySpan();

        span.setSpanId(UUIDUtils.createUUID());
        // request
        span.method = parseMethod(httpRequest);
        span.url = parseUrl(httpHost, httpRequest);
        span.requestHeaders = parseRequestHeaders(httpRequest);  // 目前该参数不参与回归的对比
        span.requestParameters = parseRequestParameters(httpRequest);
        span.requestBody = parseRequestBody(httpRequest);

        // response
        if (response != null) {
            span.setResp(response);
        }

        return span;
    }

    public void setResp(CloseableHttpResponse response) throws IOException {
        StatusLine statusLine = response.getStatusLine();
        ProtocolVersion protocolVersion = statusLine.getProtocolVersion();
        this.responseHeaders = parseResponseHeaders(response);
        this.protocol = protocolVersion.getProtocol();
        this.major = protocolVersion.getMajor();
        this.minor = protocolVersion.getMinor();
        this.statusCode = statusLine.getStatusCode();
        this.reasonPhrase = statusLine.getReasonPhrase();
        this.responseBody = EntityUtils.toByteArray(response.getEntity());
    }

    public static String parseMethod(HttpRequest httpRequest) {
        if (httpRequest instanceof HttpUriRequest) {
            HttpUriRequest httpUriRequest = (HttpUriRequest) httpRequest;
            return httpUriRequest.getMethod();
        } else if (httpRequest instanceof BasicHttpRequest) {
            BasicHttpRequest basicHttpRequest = (BasicHttpRequest) httpRequest;
            return basicHttpRequest.getRequestLine().getMethod();
        } else {
            throw new FlowReplayException("httpRequest未知实现_强转异常");
        }
    }

    public static String parseUrl(HttpHost httpHost, HttpRequest httpRequest) {
        String uriStr;
        if (httpRequest instanceof HttpUriRequest) {
            HttpUriRequest httpUriRequest = (HttpUriRequest) httpRequest;
            uriStr = httpUriRequest.getURI().toString();
        } else if (httpRequest instanceof BasicHttpRequest) {
            BasicHttpRequest basicHttpRequest = (BasicHttpRequest) httpRequest;
            uriStr = httpHost.toString() + basicHttpRequest.getRequestLine().getUri();
        } else {
            throw new FlowReplayException("httpRequest未知实现_强转异常");
        }

        if (uriStr.contains("?")) {
            uriStr = uriStr.substring(0, uriStr.indexOf("?"));
        }
        return uriStr;
    }

    public static Map<String, List<String>> parseResponseHeaders(CloseableHttpResponse response) {
        Header[] allHeaders = response.getAllHeaders();
        return parseHeaders(allHeaders);
    }

    public static Map<String, List<String>> parseRequestHeaders(HttpRequest httpRequest) {
        Header[] allHeaders = httpRequest.getAllHeaders();
        return parseHeaders(allHeaders);
    }

    private static Map<String, List<String>> parseHeaders(Header[] allHeaders) {
        Map<String, List<String>> headers = new LinkedHashMap<>();
        if (allHeaders != null && allHeaders.length > 0) {
            for (Header h : allHeaders) {
                List<String> hs = headers.computeIfAbsent(h.getName(), k -> new ArrayList<>());
                hs.add(h.getValue());
            }
        }
        return headers;
    }

    public static byte[] parseRequestBody(HttpRequest httpRequest) throws IOException {
        byte[] requestBody = null;
        if (httpRequest instanceof HttpEntityEnclosingRequest) {
            HttpEntityEnclosingRequest entityEnclosingRequest = (HttpEntityEnclosingRequest) httpRequest;
            requestBody = EntityUtils.toByteArray(entityEnclosingRequest.getEntity());
        }
        return requestBody;
    }

    public static Map<String, List<String>> parseRequestParameters(HttpRequest httpRequest) throws UnsupportedEncodingException {
        String query;
        if (httpRequest instanceof HttpUriRequest) {
            HttpUriRequest httpUriRequest = (HttpUriRequest) httpRequest;
            query = httpUriRequest.getURI().getQuery();
        } else if (httpRequest instanceof BasicHttpRequest) {
            BasicHttpRequest basicHttpRequest = (BasicHttpRequest) httpRequest;
            query = null;   // TODO
        } else {
            throw new FlowReplayException("httpRequest未知实现_强转异常");
        }
        Map<String, List<String>> parameter = new HashMap<>();
        if (StringUtils.isBlank(query)) {
            return parameter;
        }
        for (String param : query.split("&")) {
            String[] pair = param.split("=");
            String key = URLDecoder.decode(pair[0], "UTF-8");
            String value = "";
            if (pair.length > 1) {
                value = URLDecoder.decode(pair[1], "UTF-8");
            }

            List<String> values = parameter.computeIfAbsent(key, k -> new ArrayList<>());
            values.add(value);
        }
        return parameter;
    }

    public String getMethod() {
        return method;
    }

    public String getUrl() {
        return url;
    }

    public Map<String, List<String>> getRequestHeaders() {
        return requestHeaders;
    }

    public Map<String, List<String>> getRequestParameters() {
        return requestParameters;
    }

    public byte[] getRequestBody() {
        return requestBody;
    }

    public Map<String, List<String>> getResponseHeaders() {
        return responseHeaders;
    }

    public String getProtocol() {
        return protocol;
    }

    public int getMajor() {
        return major;
    }

    public int getMinor() {
        return minor;
    }

    public int getStatusCode() {
        return statusCode;
    }

    public String getReasonPhrase() {
        return reasonPhrase;
    }

    public byte[] getResponseBody() {
        return responseBody;
    }

}
