package cn.com.duibaboot.ext.autoconfigure.config;

import cn.com.duibaboot.ext.autoconfigure.flowreplay.FlowReplayUtils;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Joiner;
import org.apache.commons.lang3.StringUtils;
import org.springframework.boot.SpringApplication;
import org.springframework.boot.WebApplicationType;
import org.springframework.boot.autoconfigure.data.redis.RedisAutoConfiguration;
import org.springframework.boot.env.EnvironmentPostProcessor;
import org.springframework.core.env.ConfigurableEnvironment;
import org.springframework.core.env.MapPropertySource;
import org.springframework.core.env.PropertySource;

import java.util.*;

/**
 * 添加一些配置以禁止某些无用特性,比如servo等.<br/>
 * 配置undertow使用tomcat的配置, 如果没有，则设置undertow线程池大小为200。
 */
public class DuibaEnvironmentPostProcessor implements EnvironmentPostProcessor {

    @VisibleForTesting
    public static final Map<String, String> defProperties;
    @VisibleForTesting
    public static final String DUIBA_BOOT_DEFAULT_CONFIG = "duibaBootDefaultConfig";

    static {
        Map<String, String> overrideProperties = new HashMap<>();
        //配置禁用netflix servo监控
        overrideProperties.put("spring.metrics.servo.enabled", "false");
        overrideProperties.put("spring.metrics.export.enabled", "false");
        //启动时初始化ribbon
        overrideProperties.put("ribbon.eager-load.enabled", "true");
        //禁用一些health检查，以避免类似【elasticsearch不可用导致整个服务不可用】的问题
        overrideProperties.put("management.health.elasticsearch.enabled", "false");
        overrideProperties.put("management.health.mail.enabled", "false");
        overrideProperties.put("management.health.cassandra.enabled", "false");
        overrideProperties.put("management.health.couchbase.enabled", "false");
        overrideProperties.put("management.health.mongo.enabled", "false");
        overrideProperties.put("management.health.rabbit.enabled", "false");
        overrideProperties.put("management.health.redis.enabled", "false");
        overrideProperties.put("management.health.solr.enabled", "false");
        overrideProperties.put("management.health.hazelcast.enabled", "false");

        //强制CharacterEncodingFilter对response也编码（默认只会对request编码）
        overrideProperties.put("spring.http.encoding.force", "true");
        //让zipkin客户端使用http方式发送消息（显示指定，以防止客户端引入kafka时自动使用kafka来发送）
        overrideProperties.put("spring.zipkin.sender.type", "web");
        //不自动注册，我们自己来注册到eureka上。
        overrideProperties.put("spring.cloud.service-registry.auto-registration.enabled", "false");
        //springmvc/webflux出错时返回的消息里也带上exception字段（这个字段表示异常类的类名，在spring-cloud接口中很有用，advancedFeignClient依赖这个字段做些特殊处理）
        overrideProperties.put("server.error.include-exception", "true");
        //暴露所有endpoint到web端口上
        overrideProperties.put("management.endpoints.web.exposure.include", "*");
        //sleuth/brave使用，表示X-B3-Flags这个字段可以跨span传输
        overrideProperties.put("spring.sleuth.baggage-keys", "X-B3-Flags");
        //spring-boot2修改了jackson的默认配置，会把date类型转换为字符串输出，（1.*默认行为是输出为long），加上如下配置来保持1.*时代的设置：
        overrideProperties.put("spring.jackson.serialization.write-dates-as-timestamps", "true");

        //自动识别X-Forwarded-For and X-Forwarded-Proto头.
        overrideProperties.put("server.use-forward-headers", "true");
        //取消自动把com.zaxxer.hikari.HikariDataSource bean转化为refresh scope的动作
        overrideProperties.put("spring.cloud.refresh.refreshable", "");

        //禁止刷新eurekaClient，以防止刷新时应用从eureka服务端短暂消失导致问题。
        overrideProperties.put("eureka.client.refresh.enable", "false");

        defProperties = Collections.unmodifiableMap(overrideProperties);
    }

    @Override
    public void postProcessEnvironment(ConfigurableEnvironment environment, SpringApplication application) {
        if(application.getWebApplicationType() != WebApplicationType.REACTIVE
             && application.getWebApplicationType() != WebApplicationType.SERVLET){//判断到是bootstrap的application则跳过
            return;
        }
        Map<String, Object> properties = new HashMap<>();
        properties.putAll(defProperties);

        PropertySource<?> propertySource = new MapPropertySource(DUIBA_BOOT_DEFAULT_CONFIG,
                properties);
        //最后的优先级最低
        environment.getPropertySources().addLast(propertySource);

        configEmbedServletContainerProperties(environment, properties);
        configRedisProperties(environment, properties);
        configEurekaProperties(environment, properties);
    }

    private void configEmbedServletContainerProperties(ConfigurableEnvironment environment, Map<String, Object> properties){
        String undertowWorkerThreads = environment.getProperty("server.undertow.workerThreads");
        if(StringUtils.isBlank(undertowWorkerThreads)){
            undertowWorkerThreads = environment.getProperty("server.undertow.worker-threads");
        }

        if(!StringUtils.isBlank(undertowWorkerThreads)){
            return;
        }

        String tomcatWorkerThreads = environment.getProperty("server.tomcat.maxThreads");
        if(StringUtils.isBlank(undertowWorkerThreads)){
            tomcatWorkerThreads = environment.getProperty("server.tomcat.max-threads");
        }

        if(!StringUtils.isBlank(tomcatWorkerThreads)){//优先使用tomcat的线程数来赋值给undertow
            properties.put("server.undertow.worker-threads", tomcatWorkerThreads);
        } else {//否则配置使用默认值200
            properties.put("server.undertow.worker-threads", "200");
        }
    }

    private void configRedisProperties(ConfigurableEnvironment environment, Map<String, Object> properties){
        //禁用Spring官方的 RedisAutoConfiguration。
        String property = "spring.autoconfigure.exclude";
        String excludeAutoConfigClasses = StringUtils.defaultString(environment.getProperty(property), null);
        List<String> excludeList = new ArrayList<>();
        if(StringUtils.isNotBlank(excludeAutoConfigClasses)) {
            excludeList.add(excludeAutoConfigClasses);
        }
        excludeList.add(RedisAutoConfiguration.class.getName());
        if (FlowReplayUtils.isReplayEnv()) { //引流回归的回放实例不执行定时任务
            excludeList.add("io.elasticjob.autoconfigure.ElasticJobAutoConfiguration");
        }
        excludeList.add("org.springframework.boot.actuate.autoconfigure.hazelcast.HazelcastHealthContributorAutoConfiguration");
        //有多个redisConnectionFactory的情况下RedisReactiveAutoConfiguration会报错，先禁用
        excludeList.add("org.springframework.boot.autoconfigure.data.redis.RedisReactiveAutoConfiguration");
        excludeAutoConfigClasses = Joiner.on(",").skipNulls().join(excludeList);
        properties.put(property, excludeAutoConfigClasses);

        if (StringUtils.isBlank(environment.getProperty("duiba.redis.host"))) {
            //禁用 RedisRepositoriesAutoConfiguration
            properties.put("spring.data.redis.repositories.enabled", "false");
        }
    }

    /**
     * 如果设置了contextpath，则设置默认的eureka相关属性（会注册到eureka元数据中）
     * @param environment
     * @param properties
     */
    private void configEurekaProperties(ConfigurableEnvironment environment, Map<String, Object> properties){
        String contextPath = environment.getProperty("server.context-path");
        if(StringUtils.isBlank(contextPath)){
            contextPath = environment.getProperty("server.contextPath");
        }
        if(!StringUtils.isBlank(contextPath)) {
            properties.put("eureka.instance.home-page-url-path", contextPath);
//            properties.put("eureka.instance.status-page-url-path", contextPath + "/info");
//            properties.put("eureka.instance.health-check-url-path", contextPath + "/health");
        }
    }
}
