package cn.com.duibaboot.ext.autoconfigure.flowreplay.record.aop;

import cn.com.duibaboot.ext.autoconfigure.flowreplay.span.FlowReplayTrace;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.span.MybatisFlowReplaySpan;
import lombok.extern.slf4j.Slf4j;
import org.apache.ibatis.cache.CacheKey;
import org.apache.ibatis.executor.Executor;
import org.apache.ibatis.mapping.BoundSql;
import org.apache.ibatis.mapping.MappedStatement;
import org.apache.ibatis.plugin.*;
import org.apache.ibatis.session.ResultHandler;
import org.apache.ibatis.session.RowBounds;

import java.lang.reflect.Method;
import java.util.Properties;

/**
 * 对MyBatis进行拦截，录制
 */
@Slf4j
@Intercepts({
        @Signature(method = "query", type = Executor.class, args = { MappedStatement.class, Object.class, RowBounds.class, ResultHandler.class }),
        @Signature(method = "query", type = Executor.class, args = { MappedStatement.class, Object.class, RowBounds.class, ResultHandler.class, CacheKey.class, BoundSql.class }),
        @Signature(method = "update", type = Executor.class, args = { MappedStatement.class, Object.class }) })
public class RecordMybatisPlugin implements Interceptor {

    @Override
    public Object intercept(Invocation invocation) throws Throwable {
        if (!FlowReplayTrace.isTraced()) {
            return invocation.proceed();
        }
        // 本次调用已经被标记为需要忽略，不录制
        if (IgnoreSubInvokesContext.isMarked()) {
            return invocation.proceed();
        }

        Object target = invocation.getTarget();
        Method method = invocation.getMethod();
        Object[] args = invocation.getArgs();
        // 标记本次调用的子调用不需要录制
        IgnoreSubInvokesContext.instMark(target, method.getName(), args);

        MybatisFlowReplaySpan span = null;
        try {
            MappedStatement ms = (MappedStatement) invocation.getArgs()[0];
            Object parameter = invocation.getArgs()[1];
            // 此处先录下参数，在方法执行之后，再录下返回值，与其他的录制方式不同
            // 在执行insert之后，会回塞id，所以会影响回归流程，所以在没有回塞id的时候就录制下来
            span = MybatisFlowReplaySpan.createSpan(ms.getId(), parameter);
            span.setTraceId(FlowReplayTrace.getCurrentTraceId());
            FlowReplayTrace.addSubSpan(span);
        } catch (Throwable t) {
            log.error("Mybatis_录制异常", t);
            // 录制异常，这个用例不录了
            FlowReplayTrace.remove();
        }

        Object ret;
        try {
            ret = invocation.proceed();
        } catch (Throwable t) {
            // 如果捕捉到异常，并且正在录制中，那么不录这个用例
            FlowReplayTrace.remove();
            IgnoreSubInvokesContext.unmark();
            throw t;
        }

        try {
            if (span != null) {
                span.setRet(ret);
            }
        } catch (Throwable t) {
            log.error("Mybatis_录制异常", t);
            // 录制异常，这个用例不录了
            FlowReplayTrace.remove();
        } finally {
            IgnoreSubInvokesContext.unmark();
        }

        return ret;
    }

    @Override
    public Object plugin(Object target) {
        return Plugin.wrap(target, this);
    }

    @Override
    public void setProperties(Properties arg0) {
        // do nothing
    }

}
