package cn.com.duibaboot.ext.autoconfigure.flowreplay.record.aop;

import cn.com.duibaboot.ext.autoconfigure.flowreplay.span.FlowReplayTrace;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.FlowReplayUtils;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.span.RandomFlowReplaySpan;
import cn.com.duibaboot.ext.autoconfigure.javaagent.core.interceptor.enhance.MethodInterceptResult;
import cn.com.duibaboot.ext.autoconfigure.javaagent.core.interceptor.enhance.StaticMethodsAroundInterceptor;
import lombok.extern.slf4j.Slf4j;

import java.lang.reflect.Method;

/**
 * 随机数的方法录制
 * Created by guoyanfei .
 * 2019-05-17 .
 */
@Slf4j
public class RecordRandomMethodInterceptor implements StaticMethodsAroundInterceptor {

    /**
     * 可以录制
     * @param method
     * @param allArguments
     * @return
     */
    private boolean canRecord(Method method, Object[] allArguments) {
        if (!FlowReplayTrace.isTraced()) {
            return false;
        }
        if (FlowReplayUtils.isGlobalWhitelist(allArguments)) {
            return false;
        }
        return true;
    }

    @Override
    public void beforeMethod(Class clazz, Method method, Object[] allArguments, Class<?>[] parameterTypes, MethodInterceptResult result) {
        if (!canRecord(method, allArguments)) {
            return;
        }
        // 本次调用已经被标记为需要忽略，不录制
        if (IgnoreSubInvokesContext.isMarked()) {
            return;
        }
        // 标记本次调用的子调用不需要录制
        IgnoreSubInvokesContext.staticMark(clazz, method.getName(), allArguments);
    }

    @Override
    public Object afterMethod(Class clazz, Method method, Object[] allArguments, Class<?>[] parameterTypes, Object ret) {
        if (!canRecord(method, allArguments)) {
            return ret;
        }
        if (!IgnoreSubInvokesContext.isStaticMarked(clazz, method.getName(), allArguments)) {
            return ret;
        }
        try {
            RandomFlowReplaySpan span = RandomFlowReplaySpan.createSpan(method, allArguments, parameterTypes, ret);
            span.setTraceId(FlowReplayTrace.getCurrentTraceId());
            FlowReplayTrace.addSubSpan(span);
        } catch (Throwable t) {
            log.error("Random_录制异常", t);
            // 录制异常，这个用例不录了
            FlowReplayTrace.remove();
        } finally {
            IgnoreSubInvokesContext.unmark();
        }
        return ret;
    }

    @Override
    public void handleMethodException(Class clazz, Method method, Object[] allArguments, Class<?>[] parameterTypes, Throwable t) {
        // 如果捕捉到异常，并且正在录制中，那么不录这个用例
        if (FlowReplayTrace.isTraced()) {
            FlowReplayTrace.remove();
        }
    }
}
