package cn.com.duibaboot.ext.autoconfigure.flowreplay.replay.aop;

import cn.com.duibaboot.ext.autoconfigure.flowreplay.FlowReplayErrorMsgTypeEnum;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.FlowReplayException;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.FlowReplayUtils;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.replay.ReplayTraceContext;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.replay.StepDiffColumn;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.span.*;
import com.esotericsoftware.kryo.KryoException;
import lombok.extern.slf4j.Slf4j;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.reflect.MethodSignature;
import org.springframework.core.annotation.Order;

import java.lang.reflect.Method;
import java.util.Set;

/**
 * rocketmq回归切面
 * Created by guoyanfei .
 * 2019-05-23 .
 */
@Slf4j
@Aspect
@Order  // 默认最低优先级，这个拦截需要在最后，因为其他的拦截可能会在方法参数中加kv，如果再前面执行会导致回归参数对比不一致
public class ReplayRocketMqProducerPlugin {

    @Around("execution(* org.apache.rocketmq.client.producer.DefaultMQProducer+.*(..))")
    public Object rocketMqProducerJoinPoint(ProceedingJoinPoint joinPoint) throws Throwable {   // NOSONAR
        MethodSignature signature = (MethodSignature) joinPoint.getSignature();
        Method method = signature.getMethod();

        // 当前不是回归的环境 或者 当前请求不是回归请求
        if (!FlowReplayUtils.isReplayEnv() || !ReplayTraceContext.isReplaying() || !"send".equals(method.getName())) {
            return joinPoint.proceed();
        }

        FlowReplaySpan span = ReplayTraceContext.pollSubSpan();

        Object[] parameterValues = joinPoint.getArgs();
        // 创建回归的span，用于回归详情的构建
        RocketMqProducerFlowReplaySpan replayDetailSpan = createReplayDetailSpan(signature, parameterValues, span);

        // 增加了新的调用 || 调用内容有变动
        if (span == null || SpanType.ROCKETMQ != span.getSpanType()) {
            String expert = SpanType.ROCKETMQ.name();
            String actual = span != null ? span.getSpanType().name() : null;
            ReplayTraceContext.markError(FlowReplayErrorMsgTypeEnum.EM_001, expert, actual);
            throw new FlowReplayException(ReplayTraceContext.getCompletedErrorMsg());
        }

        RocketMqProducerFlowReplaySpan rocketMqProducerSpan = (RocketMqProducerFlowReplaySpan) span;

        // 方法名是否相同
        if (!FlowReplayUtils.isMethodEqual(rocketMqProducerSpan.getMethodName(), method)) {
            String expert = rocketMqProducerSpan.getMethodName();
            String actual = method.getName();
            ReplayTraceContext.markError(FlowReplayErrorMsgTypeEnum.EM_1001, expert, actual);
            throw new FlowReplayException(ReplayTraceContext.getCompletedErrorMsg());
        }

        // 参数类型是否相同
        if (!FlowReplayUtils.isArgumentsTypesEqual(rocketMqProducerSpan.getParameterTypes(), method.getParameterTypes())) {
            String expert = FlowReplayUtils.stringArrayToString(rocketMqProducerSpan.getParameterTypes());
            String actual = FlowReplayUtils.classArrayToString(method.getParameterTypes());
            ReplayTraceContext.markError(FlowReplayErrorMsgTypeEnum.EM_1002, expert, actual);
            throw new FlowReplayException(ReplayTraceContext.getCompletedErrorMsg());
        }

        try {
            // 对比 [录制的参数列表] 和 [回归的参数列表] ，记录对比过程中不相同的参数
            Set<StepDiffColumn> stepDiffColumns = FlowReplayUtils.compareArray(ReplayTraceContext.getContextTraceId(), ReplayTraceContext.getCurrentSpanIdx(), rocketMqProducerSpan.getParameterValues(), parameterValues);
            ReplayTraceContext.addAllStepDiffColumns(stepDiffColumns);
        } catch (KryoException e) {
            ReplayTraceContext.markError(FlowReplayErrorMsgTypeEnum.EM_1006, e);
            throw new FlowReplayException(ReplayTraceContext.getCompletedErrorMsg());
        }

        // 返回值类型是否相同
        if (!FlowReplayUtils.isReturnTypeEqual(rocketMqProducerSpan.getReturnType(), method.getReturnType())) {
            String expert = rocketMqProducerSpan.getReturnType();
            String actual = method.getReturnType() != null ? method.getReturnType().getName() : null;
            ReplayTraceContext.markError(FlowReplayErrorMsgTypeEnum.EM_1004, expert, actual);
            throw new FlowReplayException(ReplayTraceContext.getCompletedErrorMsg());
        }

        // 方法所在类全路径是否相同
        if (!FlowReplayUtils.isTypeFullPathEqual(rocketMqProducerSpan.getTypeFullPath(), signature.getDeclaringTypeName())) {
            String expert = rocketMqProducerSpan.getTypeFullPath();
            String actual = signature.getDeclaringTypeName();
            ReplayTraceContext.markError(FlowReplayErrorMsgTypeEnum.EM_1005, expert, actual);
            throw new FlowReplayException(ReplayTraceContext.getCompletedErrorMsg());
        }

        try {
            // 都相同mock返回值
            Object mockRet = rocketMqProducerSpan.getReturnValue();

            // 参数都相同的情况下，mock之前把mock的返回值回填到构建回归详情用的span中
            replayDetailSpan.setReturnValue(mockRet);

            return mockRet;
        } catch (KryoException e) {
            ReplayTraceContext.markError(FlowReplayErrorMsgTypeEnum.EM_1007, e);
            throw new FlowReplayException(ReplayTraceContext.getCompletedErrorMsg());
        }
    }

    private RocketMqProducerFlowReplaySpan createReplayDetailSpan(MethodSignature signature, Object[] parameterValues, FlowReplaySpan span) {
        RocketMqProducerFlowReplaySpan replayDetailSpan = RocketMqProducerFlowReplaySpan.createSpan(signature, parameterValues);
        replayDetailSpan.setTraceId(FlowReplayTrace.getCurrentTraceId());
        if (span != null) {
            replayDetailSpan.setSpanId(span.getSpanId());
        }
        FlowReplayTrace.addSubSpan(replayDetailSpan);
        return replayDetailSpan;
    }
}
