/*
 * Copyright (c) 2008-2017, Hazelcast, Inc. All Rights Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package cn.com.duibaboot.ext.autoconfigure.hazelcast;

import ch.qos.logback.classic.Level;
import cn.com.duiba.boot.utils.JarVersionUtils;
import cn.com.duiba.wolf.utils.NumberUtils;
import cn.com.duibaboot.ext.autoconfigure.cloud.netflix.eureka.DiscoveryMetadataAutoConfiguration;
import cn.com.duibaboot.ext.autoconfigure.cloud.netflix.eureka.EurekaClientUtils;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.FlowReplayUtils;
import com.google.common.annotations.VisibleForTesting;
import com.hazelcast.logging.ILogger;
import com.hazelcast.logging.NoLogFactory;
import com.hazelcast.nio.Address;
import com.hazelcast.spi.discovery.AbstractDiscoveryStrategy;
import com.hazelcast.spi.discovery.DiscoveryNode;
import com.hazelcast.spi.discovery.SimpleDiscoveryNode;
import com.netflix.appinfo.ApplicationInfoManager;
import com.netflix.appinfo.InstanceInfo;
import com.netflix.discovery.EurekaClient;
import com.netflix.discovery.shared.Application;
import lombok.extern.slf4j.Slf4j;

import java.net.InetAddress;
import java.net.UnknownHostException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

/**
 * hazelcast的集群发现策略，使用eureka来做服务发现
 */
@Slf4j
public final class EurekaOneDiscoveryStrategy extends AbstractDiscoveryStrategy {

    protected static final String HAZELCAST_PORT = "hazelcast.port";

    static{
        ((ch.qos.logback.classic.Logger)log).setLevel(Level.INFO);
    }

    static final class EurekaOneDiscoveryStrategyBuilder {
        private EurekaClient eurekaClient;
        private ApplicationInfoManager applicationInfoManager;
        private DiscoveryNode discoveryNode;
        private ILogger logger = new NoLogFactory().getLogger(EurekaOneDiscoveryStrategy.class.getName());
        private Map<String, Comparable> properties = Collections.emptyMap();

        EurekaOneDiscoveryStrategyBuilder setEurekaClient(final EurekaClient eurekaClient) {
            this.eurekaClient = eurekaClient;
            if (eurekaClient != null) {
                this.applicationInfoManager = eurekaClient.getApplicationInfoManager();
            }
            return this;
        }

        EurekaOneDiscoveryStrategyBuilder setDiscoveryNode(final DiscoveryNode discoveryNode) {
            this.discoveryNode = discoveryNode;
            return this;
        }

        EurekaOneDiscoveryStrategyBuilder setILogger(final ILogger logger) {
            this.logger = logger;
            return this;
        }

        EurekaOneDiscoveryStrategyBuilder setProperties(final Map<String, Comparable> properties) {
            this.properties = properties;
            return this;
        }

        EurekaOneDiscoveryStrategy build() {
            return new EurekaOneDiscoveryStrategy(this);
        }
    }

    @VisibleForTesting
    static final int NUM_RETRIES = 5;
    private static final int VERIFICATION_WAIT_TIMEOUT = 1;
    private static final int DISCOVERY_RETRY_TIMEOUT = 1;

    private final EurekaClient eurekaClient;
    private final ApplicationInfoManager applicationInfoManager;

    private EurekaOneDiscoveryStrategy(final EurekaOneDiscoveryStrategyBuilder builder) {
        super(builder.logger, builder.properties);

        this.applicationInfoManager = builder.applicationInfoManager;

        this.eurekaClient = builder.eurekaClient;
    }

    public Iterable<DiscoveryNode> discoverNodes() {//NOSONAR
        List<DiscoveryNode> nodes = new ArrayList<>();
        if(FlowReplayUtils.isReplayEnv()){
            log.info("检测到当前实例是流量回归实例，不会加入hazelcast集群!");
            return nodes;
        }

        String applicationName = applicationInfoManager.getEurekaInstanceConfig().getAppname();

        Application application = null;
        for (int i = 0; i < NUM_RETRIES; i++) {
            //获取application之前强制eureka从服务器重新获取一次
            EurekaClientUtils.refreshRegistry(eurekaClient);

            application = eurekaClient.getApplication(applicationName);
            if (application != null) {
                break;
            }
            try {
                TimeUnit.SECONDS.sleep(DISCOVERY_RETRY_TIMEOUT);
            } catch (InterruptedException almostIgnore) {
                Thread.currentThread().interrupt();
            }
        }
        if (application != null) {
            List<InstanceInfo> instances = application.getInstancesAsIsFromEureka();
            for (InstanceInfo instance : instances) {
                // Only recognize up/starting and running instances
                if (instance.getStatus() != InstanceInfo.InstanceStatus.UP
                    && instance.getStatus() != InstanceInfo.InstanceStatus.STARTING) {
                    continue;
                }

                InetAddress address = mapAddress(instance);
                if (null == address) {
                    continue;
                }

                Map<String, Object> metadata = (Map) instance.getMetadata();
                String portStr = (String)metadata.get(HAZELCAST_PORT);
                int port = NumberUtils.parseInt(portStr, -1);
                if(port == -1){//如果metadata中没有hazelcast的端口信息，则对应服务器肯定还没有接入hazelcast，忽略该服务器.
                    continue;
                }
                String duibaBootVersion = (String)metadata.get(DiscoveryMetadataAutoConfiguration.DUIBA_BOOT_VERSION);
                if(!JarVersionUtils.isJarVersionEqualOrGreaterThan(duibaBootVersion, "1.2.248")){
                    //1.2.248 之前的版本用的hazelcast版本太老，不加入这些老集群。
                    continue;
                }
                nodes.add(new SimpleDiscoveryNode(new Address(address, port), metadata));
            }
        }
        log.debug("hazelcast found nodes {}", nodes.stream().map(discoveryNode -> discoveryNode.getPrivateAddress()).collect(Collectors.toList()));
        return nodes;
    }

    @Override
    public void start() {
        if (FlowReplayUtils.isReplayEnv()) {
            return;
        }
        verifyEurekaRegistration();
    }

    @Override
    public void destroy() {
        //do nothing
    }

    private InetAddress mapAddress(InstanceInfo instance) {
        try {
            return InetAddress.getByName(instance.getIPAddr());

        } catch (UnknownHostException e) {
            getLogger().warning("InstanceInfo '" + instance + "' could not be resolved");
        }
        return null;
    }

    @VisibleForTesting
    void verifyEurekaRegistration() {
        String applicationName = applicationInfoManager.getEurekaInstanceConfig().getAppname();
        Application application;
        do {
            try {
                getLogger().info("Waiting for registration with Eureka...");
                //获取application之前强制eureka从服务器重新获取一次
                EurekaClientUtils.refreshRegistry(eurekaClient);
                application = eurekaClient.getApplication(applicationName);

                if (application != null) {
                    break;
                }
            } catch (Throwable t) {
                if (t instanceof Error) {
                    throw (Error) t;
                }
                log.error("", t);
            }

            try {
                TimeUnit.SECONDS.sleep(VERIFICATION_WAIT_TIMEOUT);
            } catch (InterruptedException almostIgnore) {
                Thread.currentThread().interrupt();
            }
        } while (true);
    }

}
