package cn.com.duibaboot.ext.autoconfigure.cloud.netflix.ribbon;

import cn.com.duibaboot.ext.autoconfigure.cloud.netflix.ribbon.loadbalancer.RuntimeRule;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.FlowReplayUtils;
import com.netflix.client.IClientConfigAware;
import com.netflix.client.config.IClientConfig;
import com.netflix.loadbalancer.*;
import org.springframework.beans.BeanUtils;
import org.springframework.boot.autoconfigure.AutoConfigureBefore;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.cloud.netflix.ribbon.PropertiesFactory;
import org.springframework.cloud.netflix.ribbon.RibbonAutoConfiguration;
import org.springframework.context.ApplicationContext;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.util.StringUtils;

import javax.annotation.Resource;
import java.lang.reflect.Constructor;

/**
 * 让ribbon使用自定义的类，包括：<br/>
 * 1.使用自定义的ping组件，每隔3秒访问服务器/monitor/check接口，如果不通或者返回不是OK，则标记为dead状态，去除对该服务器的流量（官方的只判断了从eureka取下来的状态是否UP，存在很大延迟.）<br/>
 * 2. 使用自定义的支持预热的rule组件（rule组件用于在每次调用接口时取得一个服务器），先尝试从isAlive的服务器中选一个，如果拿不到再转发到上层(上层方法默认获取全部服务器,并没有判断isAlive状态)<br/>
 * 算法：根据服务器启动时间计算权重进行轮询。
 */
@Configuration
@AutoConfigureBefore(RibbonAutoConfiguration.class)
@ConditionalOnClass({IPing.class, IRule.class, ILoadBalancer.class, IPingStrategy.class})
public class RibbonPropertiesFactoryAutoConfiguration {

    @Bean
    public PropertiesFactory propertiesFactory() {
        return new CustomPropertiesFactory();
    }

    private static class CustomPropertiesFactory extends PropertiesFactory{

        @Resource
        ApplicationContext context;

        @Override
        public String getClassName(Class clazz, String name) {
            if(clazz.equals(IRule.class)){
                return RuntimeRule.class.getName();
            }else if(clazz.equals(IPing.class)){
                // 如果是引流回归的环境，那么用默认的IPing(默认isAlive=true)
                if (FlowReplayUtils.isReplayEnv()) {
                    return DummyPing.class.getName();
                }
                return RibbonPing.class.getName();
            }else if(clazz.equals(ServerList.class)){
                return CustomRibbonServerList.class.getName();
            }
            return super.getClassName(clazz, name);
        }

        @Override
        @SuppressWarnings("unchecked")
        public <C> C get(Class<C> clazz, IClientConfig config, String name) {
            if(!ServerList.class.equals(clazz) && !IRule.class.equals(clazz)){
                return super.get(clazz, config, name);
            }
            String className = getClassName(clazz, name);
            if (StringUtils.hasText(className)) {
                try {
                    Class<?> toInstantiate = Class.forName(className);
                    return (C) instantiateWithConfig(context, toInstantiate, config);
                } catch (ClassNotFoundException e) {
                    throw new IllegalArgumentException("Unknown class to load "+className+" for class " + clazz + " named " + name);
                }
            }
            return null;
        }

        <C> C instantiateWithConfig(ApplicationContext context,
                                    Class<C> clazz, IClientConfig config) {
            C result = null;

            try {
                Constructor<C> constructor = clazz.getConstructor(IClientConfig.class);
                result = constructor.newInstance(config);
            } catch (Throwable e) {
                // Ignored
            }

            if (result == null) {
                result = BeanUtils.instantiate(clazz);

                if (result instanceof IClientConfigAware) {
                    ((IClientConfigAware) result).initWithNiwsConfig(config);
                }

                if (context != null) {
                    context.getAutowireCapableBeanFactory().autowireBean(result);
                }
//                        if(result instanceof InitializingBean){
//                            try {
//                                ((InitializingBean) result).afterPropertiesSet();
//                            } catch (Exception e) {
//                                throw new BeanCreationException("", e);
//                            }
//                        }
            }

            return result;
        }
    }

}
