package cn.com.duibaboot.ext.autoconfigure.hazelcast;

import cn.com.duiba.boot.utils.NetUtils;
import cn.com.duibaboot.ext.autoconfigure.cloud.netflix.eureka.DiscoveryMetadataRegister;
import com.hazelcast.core.HazelcastException;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Configuration;

import javax.annotation.PostConstruct;
import java.net.InetSocketAddress;
import java.net.ServerSocket;
import java.nio.channels.ServerSocketChannel;
import java.util.Map;
import java.util.concurrent.TimeUnit;

/**
 * 配置hazelcast的端口和eureka元数据注册器
 */
@Configuration
@Slf4j
public class DuibaHazelcastPortConfiguration implements DiscoveryMetadataRegister {

    private static final int SOCKET_TIMEOUT_MILLIS = (int) TimeUnit.SECONDS.toMillis(1);

    @Value("${server.port}")
    private int httpServerPort;

    @Value("${duiba.hazelcast.port:5701}")
    private int hazelcastScanStartPort;//开始扫描的端口

    private int hazelcastPort;//port

    public int getHazelcastPort() {
        return hazelcastPort;
    }

    //从5701到5801这100个端口扫描出一个本地可用的端口作为hazelcast的端口
    @PostConstruct
    public void init(){
        Throwable error = null;
        InetSocketAddress inetSocketAddress;

        boolean reuseAddress = true;

        int portCount = 100;
        int startScanPort = hazelcastScanStartPort;
        int port = startScanPort;
        int portDecidedToUse = -1;
        String localIp = NetUtils.getLocalIp();

        for (int i = 0; i < portCount; i++) {
            if(httpServerPort == port){//跳过http端口
                port++;
            }
            /**
             * Instead of reusing the ServerSocket/ServerSocketChannel, we are going to close and replace them on
             * every attempt to find a free port. The reason to do this is because in some cases, when concurrent
             * threads/processes try to acquire the same port, the ServerSocket gets corrupted and isn't able to
             * find any free port at all (no matter if there are more than enough free ports available). We have
             * seen this happening on Linux and Windows environments.
             */
            try(ServerSocketChannel serverSocketChannel = ServerSocketChannel.open();
                ServerSocket serverSocket = serverSocketChannel.socket();
                ) {
                serverSocket.setReuseAddress(reuseAddress);
                serverSocket.setSoTimeout(SOCKET_TIMEOUT_MILLIS);

                inetSocketAddress = new InetSocketAddress(localIp, port);
                log.debug("Trying to bind inet socket address:" + inetSocketAddress);
                serverSocket.bind(inetSocketAddress);
                log.info("Hazelcast will use port: {}", port);
                portDecidedToUse = port;
                break;
            } catch (Exception e) {
                port++;
                error = e;
            }
        }
        if (portDecidedToUse == -1) {
            throw new HazelcastException("ServerSocket bind has failed. Hazelcast cannot start!"
                    + " config-port: " + startScanPort + ", latest-port: " + port, error);
        }

        hazelcastPort = portDecidedToUse;
    }

    /**
     * 注册hazelcast本机端口到eureka中
     * @return
     */
    @Override
    public void registerMetadata(Map<String, String> appMetadata) {
        appMetadata.put(EurekaOneDiscoveryStrategy.HAZELCAST_PORT, String.valueOf(hazelcastPort));
    }
}
